package com.elitescloud.boot.auth.provider.security.generator.token;

import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.model.OAuthToken;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.core.AuthorizationGrantType;
import org.springframework.security.oauth2.core.OAuth2AccessToken;
import org.springframework.security.oauth2.jose.jws.SignatureAlgorithm;
import org.springframework.security.oauth2.jwt.*;
import org.springframework.security.oauth2.server.authorization.OAuth2TokenType;
import org.springframework.security.oauth2.server.authorization.token.JwtEncodingContext;
import org.springframework.security.oauth2.server.authorization.token.OAuth2TokenCustomizer;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;

/**
 * jwt token生成器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/10
 */
@Log4j2
public class JwtTokenGenerator implements TokenGenerator {
    private static final String ISSUER_DEFAULT = "cloudt";
    private final AuthorizationProperties authorizationProperties;
    private final JwtEncoder jwtEncoder;
    private OAuth2TokenCustomizer<JwtEncodingContext> tokenCustomizer;

    public JwtTokenGenerator(AuthorizationProperties authorizationProperties, JwtEncoder jwtEncoder) {
        this.authorizationProperties = authorizationProperties;
        this.jwtEncoder = jwtEncoder;
    }

    @Override
    public OAuthToken generate(Authentication authentication) {
        var jwt = generateJwt(authentication);
        return convertToken(jwt);
    }

    public void setTokenCustomizer(OAuth2TokenCustomizer<JwtEncodingContext> tokenCustomizer) {
        this.tokenCustomizer = tokenCustomizer;
    }

    /**
     * 生成jwt
     *
     * @param authentication authentication
     * @return jwt
     */
    private Jwt generateJwt(Authentication authentication) {
        Instant issuedAt = Instant.now();

        // claimBuilder
        var claimBuilder = JwtClaimsSet.builder()
                .issuer(ISSUER_DEFAULT)
                .subject(authentication.getName())
                .audience(Collections.singletonList(authentication.getName()))
                .issuedAt(issuedAt);

        // 设置有效期
        Duration ttl = cachePrincipalDuration();
        if (ttl != null) {
            claimBuilder.expiresAt(issuedAt.plus(ttl));
        }

        JwsHeader.Builder headersBuilder = JwsHeader.with(SignatureAlgorithm.RS256);
        JwtEncodingContext.Builder jwtContextBuilder = JwtEncodingContext.with(headersBuilder, claimBuilder)
                .principal(authentication)
                .tokenType(OAuth2TokenType.ACCESS_TOKEN)
                .authorizationGrantType(AuthorizationGrantType.JWT_BEARER);

        JwtEncodingContext jwtContext = jwtContextBuilder.build();
        // 自定义claim
        if (tokenCustomizer != null) {
            tokenCustomizer.customize(jwtContext);
        }

        JwsHeader headers = headersBuilder.build();
        JwtClaimsSet claims = claimBuilder.build();

        return jwtEncoder.encode(JwtEncoderParameters.from(headers, claims));
    }

    /**
     * jwt转自定义token格式
     *
     * @param jwt jwt
     * @return token
     */
    private OAuthToken convertToken(Jwt jwt) {
        OAuthToken token = new OAuthToken();
        token.setAccessToken(jwt.getTokenValue());
        // 目前使用bearer
        token.setTokenType(OAuth2AccessToken.TokenType.BEARER.getValue());

        // 设置过期时间
        if (jwt.getExpiresAt() == null) {
            token.setExpiresIn(-1L);
        } else {
            long expiresIn = ChronoUnit.SECONDS.between(Instant.now(), jwt.getExpiresAt());
            token.setExpiresIn(expiresIn);
        }

        token.setScope(Collections.emptySet());

        // 暂时不用refreshToken
        token.setRefreshToken(null);

        return token;
    }

    protected Duration cachePrincipalDuration() {
        if (authorizationProperties.getTokenTtl() != null && authorizationProperties.getTokenTtl().getSeconds() > 0) {
            return authorizationProperties.getTokenTtl();
        }
        return null;
    }
}
