package com.elitescloud.boot.auth.provider.security.grant;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.common.LoginType;
import com.elitescloud.boot.auth.client.config.security.handler.DelegateAuthenticationCallable;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.provider.user.UserDetailManager;
import com.elitescloud.boot.auth.provider.security.AuthenticationCheckService;
import com.elitescloud.boot.auth.provider.security.generator.token.TokenGenerator;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AccountExpiredException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.authentication.LockedException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 内部认证授权.
 * <p>
 * 对于内部已认证授权过的应用
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/19
 */
@Log4j2
public class InternalAuthenticationGranter {

    private final UserDetailManager userDetailManager;
    private final TokenGenerator tokenGenerator;
    private UserDetailsChecker userDetailsChecker = this::defaultAuthenticationChecker;
    private List<AuthenticationCheckService> authenticationCheckService;
    private DelegateAuthenticationCallable delegateAuthenticationCallable;

    public InternalAuthenticationGranter(UserDetailManager userDetailManager, TokenGenerator tokenGenerator) {
        this.userDetailManager = userDetailManager;
        this.tokenGenerator = tokenGenerator;
    }

    /**
     * 内部认证并生成token
     *
     * @param authenticationToken
     * @return token
     */
    public OAuthToken authenticate(HttpServletRequest request, HttpServletResponse response, @NotNull InternalAuthenticationToken authenticationToken) {
        AuthenticationException exception = null;
        InternalAuthenticationToken authentication = null;
        OAuthToken token = null;
        try {
            authentication = this.load(authenticationToken);
            token = this.tokenGenerator.generate(authentication);;
        } catch (AuthenticationException e) {
            exception = e;
        }

        // 登录成功后的回调
        if (delegateAuthenticationCallable != null) {
            try {
                if (exception == null) {
                    delegateAuthenticationCallable.onLogin(request, response, token.getAccessToken(), authentication);
                } else {
                    delegateAuthenticationCallable.onLoginFailure(request, response, authenticationToken, exception);
                }
            } catch (Exception e) {
                log.info("认证回调异常", e);
                if (exception == null) {
                    return token;
                }
                throw exception;
            }
        }
        return token;
    }

    private InternalAuthenticationToken load(InternalAuthenticationToken authenticationToken) {
        var userDetails = loadUser(authenticationToken.getIdType(), authenticationToken.getId());

        // 默认检查
        userDetailsChecker.check(userDetails);
        // 其它检查
        for (AuthenticationCheckService checkService : authenticationCheckService) {
            checkService.additionalAuthenticationChecks(userDetails, authenticationToken);
        }

        // 生成token
        return new InternalAuthenticationToken(authenticationToken.getIdType(), authenticationToken.getId(), userDetails, Collections.emptyList());
    }

    @Autowired
    public void setAuthenticationCheckServiceObjectProvider(ObjectProvider<AuthenticationCheckService> authenticationCheckServiceObjectProvider) {
        this.authenticationCheckService = authenticationCheckServiceObjectProvider.stream().collect(Collectors.toList());
    }

    public void setDelegateAuthenticationCallable(DelegateAuthenticationCallable delegateAuthenticationCallable) {
        this.delegateAuthenticationCallable = delegateAuthenticationCallable;
    }

    private GeneralUserDetails loadUser(IdType idType, String id) {
        Assert.hasText(id, "加载用户失败，ID为空");
        Assert.notNull(idType, "加载用户失败，ID类型为空");

        switch (idType) {
            case USER_ID:
                return userDetailManager.loadUserById(id);
            case USERNAME:
                return userDetailManager.loadUserByUsername(id);
            case EMAIL:
                return userDetailManager.loadUserByEmail(id);
            case MOBILE:
                return userDetailManager.loadUserByMobile(id);
            default:
                throw new AuthorizationException("认证失败，暂不支持的账号类型");
        }
    }

    private void defaultAuthenticationChecker(UserDetails user) {
        if (user == null) {
            throw new UsernameNotFoundException("账号不存在");
        }
        if (!user.isAccountNonLocked()) {
            log.debug("Failed to authenticate since user account is locked");
            throw new LockedException("账号已锁定");
        }
        if (!user.isEnabled()) {
            log.debug("Failed to authenticate since user account is disabled");
            throw new DisabledException("账号已禁用");
        }
        if (!user.isAccountNonExpired()) {
            log.debug("Failed to authenticate since user account has expired");
            throw new AccountExpiredException("账号已过期");
        }
    }

    public enum IdType {
        /**
         * 用户ID
         */
        USER_ID,
        /**
         * 登录账号
         */
        USERNAME,
        /**
         * 手机号
         */
        MOBILE,
        /**
         * 邮箱
         */
        EMAIL
    }

    public static final class InternalAuthenticationToken extends AbstractCustomAuthenticationToken<InternalAuthenticationToken> {
        private static final long serialVersionUID = -6577165524964905618L;
        private final IdType idType;
        private final String id;

        @Override
        public LoginType loginType() {
            return LoginType.INTERNAL;
        }

        @Override
        public InternalAuthenticationToken convert(HttpServletRequest request) {
            return null;
        }

        public InternalAuthenticationToken() {
            super(null, null);
            this.idType = null;
            this.id = null;
        }

        public InternalAuthenticationToken(IdType idType, String id) {
            super(null, null);
            this.idType = idType;
            this.id = id;

            // 未授权
            super.setAuthenticated(false);
        }

        public InternalAuthenticationToken(IdType idType, String id, Object principal, Collection<GrantedAuthority> authorities) {
            super(principal, null, authorities);
            this.idType = idType;
            this.id = id;

            // 已授权
            super.setAuthenticated(true);
        }

        public IdType getIdType() {
            return idType;
        }

        public String getId() {
            return id;
        }
    }
}
