package com.elitescloud.boot.auth.provider.security.grant.account_pwd;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.provider.LoginSupportProvider;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.util.StringUtils;

/**
 * 账号密码的身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
@Log4j2
public class AccountPasswordAuthenticationProvider extends AbstractCustomAuthenticationProvider<AccountPasswordAuthenticationToken> {

    @Autowired
    private LoginSupportProvider loginSupportManager;

    @NonNull
    @Override
    protected GeneralUserDetails retrieveUser(AccountPasswordAuthenticationToken authentication) throws AuthenticationException {
        String account = (String) authentication.getPrincipal();
        if (!StringUtils.hasText(account)) {
            throw new AuthorizationException("账号为空");
        }

        // 检查验证码
        checkCaptcha(authentication);

        var user =  userDetailManager.loadUserByAccount(account);
        if (user == null) {
            throw new UsernameNotFoundException("账号或密码错误");
        }
        return user;
    }

    @Override
    protected void additionalAuthenticationChecks(GeneralUserDetails userDetails, AccountPasswordAuthenticationToken authentication) throws AuthenticationException {
        super.additionalAuthenticationChecks(userDetails, authentication);

        if (authentication.getCredentials() == null) {
            log.debug("Failed to authenticate since no credentials provided");
            throw new BadCredentialsException("账号或密码不正确");
        }
        String presentedPassword = authentication.getCredentials().toString();
        if (!this.passwordEncoder.matches(presentedPassword, userDetails.getPassword())) {
            log.debug("Failed to authenticate since password does not match stored value");
            throw new BadCredentialsException("账号或密码不正确");
        }
    }

    private void checkCaptcha(AccountPasswordAuthenticationToken authentication) {
        var need = authentication.getCaptchaNeeded();
        if (need != null && !need) {
            // 无需校验
            return;
        }
        var verifyResult = loginSupportManager.verifyCaptcha(authentication.getCaptchaKey(), authentication.getCaptchaText());
        if (!verifyResult.isSuccess()) {
            // 校验不通过时
            throw new AuthorizationException(verifyResult.getMsg());
        }

        if (verifyResult.getData() == null || !verifyResult.getData()) {
            throw new AuthorizationException("验证码已过期或验证码错误");
        }
    }
}
