package com.elitescloud.boot.auth.provider.security.grant.wechat_minapp;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.provider.wechat.WechatProvider;
import com.elitescloud.boot.auth.provider.provider.wechat.param.WechatUserInfo;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.util.StringUtils;

/**
 * 微信openid的身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
@Log4j2
public class WechatMinAppAuthenticationProvider extends AbstractCustomAuthenticationProvider<WechatMinAppAuthenticationToken> {

    @Autowired
    private WechatProvider wechatProvider;

    @Override
    protected GeneralUserDetails retrieveUser(WechatMinAppAuthenticationToken authentication) throws AuthenticationException {
        String openId = authentication.getOpenId();
        String appId = (String) authentication.getPrincipal();
        String wxCode = (String) authentication.getCredentials();

        if (StringUtils.hasText(openId)) {
            // 有openId，则直接加载用户信息
            var user = userDetailManager.loadUserByWechatOpenid(openId);
            if (user == null) {
                throw new UsernameNotFoundException("该微信未绑定账号");
            }
            return user;
        }

        // 没有openId，则根据授权码换取openId
        if (!StringUtils.hasText(appId) || !StringUtils.hasText(wxCode)) {
            throw new AuthorizationException("请求有误，参数不完整");
        }

        // 调用微信接口，根据wxCode获得用户的openId
        openId = obtainOpenid(appId, wxCode);
        var user = userDetailManager.loadUserByWechatOpenid(openId);
        if (user == null) {
            throw new UsernameNotFoundException("该微信未绑定账号");
        }
        return user;
    }

    private String obtainOpenid(String appId, String wxCode) {
        try {
            WechatUserInfo userInfo = wechatProvider.getUserInfo(appId, wxCode);
            if (userInfo != null) {
                return userInfo.getOpenid();
            }
        } catch (Exception e) {
            throw new AuthorizationException("微信服务授权异常", e);
        }

        return null;
    }
}
