package com.elitescloud.boot.auth.provider.config;

import com.elitescloud.boot.auth.client.config.security.AbstractServletSecurityConfig;
import com.elitescloud.boot.auth.client.config.security.handler.DelegateAuthenticationCallable;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCallable;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.provider.provider.user.UserDetailManager;
import com.elitescloud.boot.auth.provider.security.generator.ticket.RedisTicketGenerator;
import com.elitescloud.boot.auth.provider.security.generator.ticket.TicketGenerator;
import com.elitescloud.boot.auth.provider.security.generator.token.TokenGenerator;
import com.elitescloud.boot.auth.provider.security.handler.sso.SsoTicketAuthenticationCallable;
import com.elitescloud.boot.auth.provider.sso.SsoLoginSupportController;
import com.elitescloud.boot.auth.provider.sso.SsoLoginSupportProvider;
import com.elitescloud.boot.auth.provider.sso.impl.CloudtSsoLoginSupportProviderImpl;
import com.elitescloud.boot.auth.provider.sso.impl.CloudtSsoProviderImpl;
import com.elitescloud.boot.auth.sso.TicketResolver;
import com.elitescloud.boot.auth.sso.impl.CookieTicketResolver;
import lombok.extern.log4j.Log4j2;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;

/**
 * 单点登录相关配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/15
 */
@ConditionalOnProperty(prefix = AuthorizationSdkProperties.CONFIG_PREFIX + ".sso", name = "enabled", havingValue = "true")
@Log4j2
class SsoConfig extends AbstractServletSecurityConfig {

    private final AuthorizationSdkProperties sdkProperties;

    public SsoConfig(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
        log.info("启用单点登录功能");
    }

    @Bean
    @ConditionalOnMissingBean
    private CloudtSsoProviderImpl cloudtSsoProvider(TokenGenerator tokenGenerator
            , AuthenticationCache authenticationCache
            , UserDetailManager userDetailManager
            , RedisHelper redisHelper) {
        CloudtSsoProviderImpl provider = new CloudtSsoProviderImpl(tokenGenerator, authenticationCache, userDetailManager, redisHelper);
        provider.setSdkProperties(sdkProperties);
        provider.setAuthenticationCallable(DelegateAuthenticationCallable.getInstance());
        return provider;
    }

    @Bean
    public SsoLoginSupportController ssoLoginSupportController(SsoLoginSupportProvider ssoLoginSupportProvider) {
        return new SsoLoginSupportController(ssoLoginSupportProvider);
    }

    @Bean
    @ConditionalOnMissingBean
    public SsoLoginSupportProvider ssoLoginSupportProvider(TicketGenerator ticketGenerator,
                                                           UserDetailManager userDetailManager) {
        return new CloudtSsoLoginSupportProviderImpl(sdkProperties, ticketGenerator, userDetailManager);
    }

    /**
     * 登录成功后生成ticket
     *
     * @param ticketGenerator
     * @param ticketResolver
     * @return
     */
    @Bean
    public AuthenticationCallable authenticationCallableSsoTicket(TicketGenerator ticketGenerator,
                                                                  TicketResolver ticketResolver) {
        return new SsoTicketAuthenticationCallable(sdkProperties, ticketGenerator, ticketResolver);
    }

    /**
     * ticket处理器
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    public TicketResolver ticketResolverCookie() {
        return new CookieTicketResolver(sdkProperties);
    }

    /**
     * redis ticket生成器
     *
     * @param redisHelper
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnBean(RedisHelper.class)
    public TicketGenerator ticketGenerator(RedisHelper redisHelper) {
        return new RedisTicketGenerator(redisHelper);
    }
}
