package com.elitescloud.boot.auth.provider.security.grant.email_pwd;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.provider.LoginSupportProvider;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.boot.auth.provider.security.jackson.mixin.grant.MixinEmailPasswordAuthenticationToken;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.StringUtils;

/**
 * 邮箱密码认证.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/22
 */
@Slf4j
public class EmailPasswordAuthenticationProvider extends AbstractCustomAuthenticationProvider<EmailPasswordAuthenticationToken> {

    @Autowired
    private LoginSupportProvider loginSupportManager;

    @NonNull
    @Override
    protected GeneralUserDetails retrieveUser(EmailPasswordAuthenticationToken authentication) throws AuthenticationException {
        String email = (String) authentication.getPrincipal();
        if (!StringUtils.hasText(email)) {
            throw new AuthorizationException("邮箱为空");
        }

        // 检查验证码
        checkCaptcha(authentication);

        var user = userDetailManager.loadUserByEmail(email);
        if (user == null) {
            throw new AuthorizationException("邮箱或密码错误");
        }
        return user;
    }

    @Override
    protected void additionalAuthenticationChecks(GeneralUserDetails userDetails, EmailPasswordAuthenticationToken authentication) throws AuthenticationException {
        super.additionalAuthenticationChecks(userDetails, authentication);

        // 是否需要校验密码
        if (!super.credentialCheckable.needCheck(authentication, userDetails)) {
            return;
        }

        if (authentication.getCredentials() == null) {
            log.debug("Failed to authenticate since no credentials provided");
            throw new BadCredentialsException("邮箱或密码不正确");
        }
        String presentedPassword = authentication.getCredentials().toString();
        if (!this.passwordEncoder.matches(presentedPassword, userDetails.getPassword())) {
            log.debug("Failed to authenticate since password does not match stored value");
            throw new BadCredentialsException("邮箱或密码不正确");
        }
    }

    @Override
    public Class<EmailPasswordAuthenticationToken> getAuthenticationTokenType() {
        return EmailPasswordAuthenticationToken.class;
    }

    @Override
    public Class<?> getMixinAuthenticationTokenType() {
        return MixinEmailPasswordAuthenticationToken.class;
    }

    private void checkCaptcha(EmailPasswordAuthenticationToken authentication) {
        var need = authentication.getCaptchaNeeded();
        if (need != null && !need) {
            // 无需校验
            return;
        }
        var verifyResult = loginSupportManager.verifyCaptcha(authentication.getCaptchaKey(), authentication.getCaptchaText());
        if (!verifyResult.isSuccess()) {
            // 校验不通过时
            throw new AuthorizationException(verifyResult.getMsg());
        }

        if (verifyResult.getData() == null || !verifyResult.getData()) {
            throw new AuthorizationException("验证码已过期或验证码错误");
        }
    }
}
