package com.elitescloud.boot.auth.provider.security.grant.mobile_pwd;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.provider.LoginSupportProvider;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.boot.auth.provider.security.grant.password.UsernamePasswordAuthenticationToken;
import com.elitescloud.boot.auth.provider.security.jackson.mixin.grant.MixinMobilePasswordAuthenticationToken;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.StringUtils;

/**
 * 手机号和密码的身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
@Log4j2
public class MobilePasswordAuthenticationProvider extends AbstractCustomAuthenticationProvider<MobilePasswordAuthenticationToken> {

    @Autowired
    private LoginSupportProvider loginSupportManager;

    @NonNull
    @Override
    protected GeneralUserDetails retrieveUser(MobilePasswordAuthenticationToken authentication) throws AuthenticationException {
        String mobile = (String) authentication.getPrincipal();
        if (!StringUtils.hasText(mobile)) {
            throw new AuthorizationException("手机号为空");
        }

        // 检查验证码
        checkCaptcha(authentication);

        var user = userDetailManager.loadUserByMobile(mobile);
        if (user == null) {
            throw new AuthorizationException("手机号或密码错误");
        }
        return user;
    }

    @Override
    protected void additionalAuthenticationChecks(GeneralUserDetails userDetails, MobilePasswordAuthenticationToken authentication) throws AuthenticationException {
        super.additionalAuthenticationChecks(userDetails, authentication);

        // 是否需要校验密码
        if (!super.credentialCheckable.needCheck(authentication, userDetails)) {
            return;
        }

        if (authentication.getCredentials() == null) {
            log.debug("Failed to authenticate since no credentials provided");
            throw new BadCredentialsException("手机号或密码不正确");
        }
        String presentedPassword = authentication.getCredentials().toString();
        if (!this.passwordEncoder.matches(presentedPassword, userDetails.getPassword())) {
            log.debug("Failed to authenticate since password does not match stored value");
            throw new BadCredentialsException("手机号或密码不正确");
        }
    }

    @Override
    public Class<MobilePasswordAuthenticationToken> getAuthenticationTokenType() {
        return MobilePasswordAuthenticationToken.class;
    }

    @Override
    public Class<?> getMixinAuthenticationTokenType() {
        return MixinMobilePasswordAuthenticationToken.class;
    }

    private void checkCaptcha(MobilePasswordAuthenticationToken authentication) {
        var need = authentication.getCaptchaNeeded();
        if (need != null && !need) {
            // 无需校验
            return;
        }
        var verifyResult = loginSupportManager.verifyCaptcha(authentication.getCaptchaKey(), authentication.getCaptchaText());
        if (!verifyResult.isSuccess()) {
            // 校验不通过时
            throw new AuthorizationException(verifyResult.getMsg());
        }

        if (verifyResult.getData() == null || !verifyResult.getData()) {
            throw new AuthorizationException("验证码已过期或验证码错误");
        }
    }
}
