package com.elitescloud.boot.auth.provider.security.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.common.LoginDeviceLimitStrategy;
import com.elitescloud.boot.auth.provider.common.LoginParameterNames;
import com.elitescloud.boot.auth.provider.common.param.UserLoginDeviceDTO;
import com.elitescloud.boot.auth.provider.config.properties.AuthorizationProviderProperties;
import com.elitescloud.boot.auth.provider.config.system.LoginProperties;
import com.elitescloud.boot.auth.provider.security.AuthenticationCheckService;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysUserType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.AuthenticationException;

import javax.servlet.http.HttpServletRequest;
import java.util.List;

/**
 * 默认的用户认证信息检查.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/20
 */
public class DefaultAuthenticationCheckServiceImpl<T extends AbstractCustomAuthenticationToken<T>> implements AuthenticationCheckService<T> {
    private static final Logger logger = LoggerFactory.getLogger(DefaultAuthenticationCheckServiceImpl.class);

    private final AuthorizationProviderProperties configProperties;
    private final AuthenticationCache authenticationCache;

    public DefaultAuthenticationCheckServiceImpl(AuthorizationProviderProperties configProperties,
                                                 AuthenticationCache authenticationCache) {
        this.configProperties = configProperties;
        this.authenticationCache = authenticationCache;
    }

    @Override
    public void additionalAuthenticationChecks(GeneralUserDetails userDetails, T authentication) throws AuthenticationException {
        // 账户类型校验
        validateUserType(userDetails);

        // 终端校验
        validateTerminal(userDetails, authentication);

        // 登录设备校验
        validateDevice(userDetails, authentication);
    }

    /**
     * 账号类型校验
     *
     * @param userDetails
     */
    private void validateUserType(GeneralUserDetails userDetails) {
        HttpServletRequest request = HttpServletUtil.currentRequest();
        if (request == null) {
            return;
        }

        var userType = request.getParameter(LoginParameterNames.USER_TYPE);
        logger.info("loginLimit-userType：{}", userType);
        if (StrUtil.isBlank(userType)) {
            return;
        }
        if (CollUtil.isNotEmpty(userDetails.getUser().getUserTypeList())) {
            for (SysUserType sysUserType : userDetails.getUser().getUserTypeList()) {
                if (userType.equals(sysUserType.getUserType())) {
                    return;
                }
            }
        }

        throw new AuthorizationException("无权限登录");
    }

    /**
     * 登录设备校验
     *
     * @param userDetails
     * @param authentication
     */
    private void validateDevice(GeneralUserDetails userDetails, T authentication) {
        HttpServletRequest request = HttpServletUtil.currentRequest();
        if (request == null) {
            return;
        }

        // 限制的设备
        var limiters = configProperties.getLogin().getLoginDeviceLimiters();
        if (CollUtil.isEmpty(limiters)) {
            return;
        }

        // 当前设备
        String clientId = (String) request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID);
        if (CharSequenceUtil.isBlank(clientId)) {
            return;
        }

        // 限制策略
        LoginProperties.LoginDeviceLimiter limiter = null;
        for (LoginProperties.LoginDeviceLimiter l : limiters) {
            if (clientId.equals(l.getClientId())) {
                limiter = l;
                break;
            }
        }
        if (limiter == null || LoginDeviceLimitStrategy.NO_LIMIT == limiter.getStrategy()) {
            return;
        }
        if (LoginDeviceLimitStrategy.ONE_LIMIT == limiter.getStrategy()) {
            List<UserLoginDeviceDTO> deviceList = (List<UserLoginDeviceDTO>) authenticationCache.getAttribute(userDetails.getUserId().toString() + SecurityConstants.CACHE_SUFFIX_CURRENT_USER_ATTRIBUTE_DEVICE);
            if (CollUtil.isEmpty(deviceList)) {
                return;
            }
            for (UserLoginDeviceDTO userLoginDeviceDTO : deviceList) {
                if (clientId.equals(userLoginDeviceDTO.getClientId())) {
                    if (authenticationCache.exists(userLoginDeviceDTO.getToken())) {
                        throw new AuthorizationException(CharSequenceUtil.blankToDefault(limiter.getOneLimitTip(), "您已在其它设备登录"));
                    }
                }
            }
        }
    }

    private void validateTerminal(GeneralUserDetails userDetails, T authentication) {
        Boolean terminalEnabled = configProperties.getLogin().getTerminalLimit();
        if (!ObjectUtil.defaultIfNull(terminalEnabled, false)) {
            // 默认不校验
            return;
        }

        Terminal terminal = authentication.getTerminal();
        if (terminal == null) {
            throw new AuthorizationException("未设置登录终端");
        }

        var terminals = userDetails.getUser().getTerminals();
        if (terminals == null || terminals.isEmpty()) {
            throw new AuthorizationException("无权限登录该终端");
        }

        if (!terminals.contains(terminal)) {
            throw new AuthorizationException("无权限登录该终端");
        }
    }
}
