package com.elitescloud.boot.auth.provider.security.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.auth.client.config.security.handler.DelegateAuthenticationCallable;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.AuthenticationService;
import com.elitescloud.boot.auth.provider.common.LoginParameterNames;
import com.elitescloud.boot.auth.provider.common.param.UserLoginDeviceDTO;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.security.handler.LogoutRedirectHandler;
import com.elitescloud.boot.auth.provider.security.listener.HttpSessionHolder;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.util.ArrayUtil;
import com.elitescloud.boot.util.StrUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.rememberme.AbstractRememberMeServices;
import org.springframework.util.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/11
 */
@Log4j2
public class DefaultAuthenticationService implements AuthenticationService {
    public static final String DEFAULT_SESSION_ID = "JSESSIONID";

    private final DelegateAuthenticationCallable authenticationCallable;
    private final LogoutRedirectHandler logoutRedirectHandler;
    private final AuthenticationCache authenticationCache;

    public DefaultAuthenticationService(DelegateAuthenticationCallable authenticationCallable, LogoutRedirectHandler logoutRedirectHandler,
                                        InternalAuthenticationGranter internalAuthenticationGranter, AuthenticationCache authenticationCache) {
        this.authenticationCallable = authenticationCallable;
        this.logoutRedirectHandler = logoutRedirectHandler;
        this.authenticationCache = authenticationCache;
    }

    @Override
    public ApiResult<OAuthToken> attempt(HttpServletRequest request, HttpServletResponse response) {
        var cookies = request.getCookies();
        String cookieValue = null;
        if (ArrayUtil.isNotEmpty(cookies)) {
            for (var cookie : cookies) {
                if (AbstractRememberMeServices.SPRING_SECURITY_REMEMBER_ME_COOKIE_KEY.equals(cookie.getName())) {
                    cookieValue = cookie.getValue();
                    break;
                }
            }
        }
        log.info("Cookie：{}", cookieValue);
        return ApiResult.fail("请登录");
    }

    @Override
    public ApiResult<String> logout(HttpServletRequest request, HttpServletResponse response) {
        String token = SecurityContextUtil.currentToken();
        GeneralUserDetails user = SecurityContextUtil.currentUser();

        // 获取重定向路径
        String redirectUrl = request.getParameter(LoginParameterNames.REDIRECT_URL);
        if (CharSequenceUtil.isBlank(redirectUrl)) {
            redirectUrl = logoutRedirectHandler == null ? null : logoutRedirectHandler.determineUrlToUseForThisRequest(request, response);
        }

        String username = null;
        if (StringUtils.hasText(token) && user != null) {
            username = user.getUsername();
        }
        // 清空session
        clearSession(request, response);
        // 回调
        authenticationCallable.onLogout(request, response, token, user);

        if (StringUtils.hasText(redirectUrl)) {
            // 重定向
            try {
                response.sendRedirect(redirectUrl);
                return null;
            } catch (IOException e) {
                log.error(redirectUrl + "重定向异常：", e);
            }
            return ApiResult.fail();
        }

        return ApiResult.ok(username);
    }

    @Override
    public ApiResult<GeneralUserDetails> currentUser() {
        GeneralUserDetails user = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        user.getUser().setPassword("");

        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<List<UserLoginDeviceDTO>> userOnlineDevice(Long userId) {
        if (userId == null) {
            return ApiResult.fail("账户ID为空");
        }

        List<UserLoginDeviceDTO> deviceList = queryUserDeviceOnline(userId);
        return ApiResult.ok(deviceList);
    }

    @Override
    public ApiResult<Boolean> expiredUserOnlineDevice(Long userId, String clientId) {
        if (userId == null) {
            return ApiResult.fail("账户ID为空");
        }

        List<UserLoginDeviceDTO> deviceList = queryUserDeviceOnline(userId);
        if (deviceList.isEmpty()) {
            return ApiResult.ok(true);
        }

        for (UserLoginDeviceDTO userLoginDeviceDTO : deviceList) {
            if (StrUtil.isBlank(clientId) || clientId.equals(userLoginDeviceDTO.getClientId())) {
                authenticationCache.removeUserDetail(userLoginDeviceDTO.getToken());
            }
        }
        return ApiResult.ok(true);
    }

    private List<UserLoginDeviceDTO> queryUserDeviceOnline(Long userId) {
        List<UserLoginDeviceDTO> deviceList = (List<UserLoginDeviceDTO>) authenticationCache.getAttribute(userId.toString() + SecurityConstants.CACHE_SUFFIX_CURRENT_USER_ATTRIBUTE_DEVICE);
        if (CollUtil.isEmpty(deviceList)) {
            return Collections.emptyList();
        }

        // 过滤出有效的
        return deviceList.stream().filter(t -> authenticationCache.exists(t.getToken())).collect(Collectors.toList());
    }

    private void clearSession(HttpServletRequest request, HttpServletResponse response) {
        HttpSession session = null;
        // 优先取自定义的
        var customSessionId = CharSequenceUtil.blankToDefault(request.getParameter(SecurityConstants.HEADER_SESSION_ID), request.getHeader(SecurityConstants.HEADER_SESSION_ID));
        if (StringUtils.hasText(customSessionId)) {
            session = HttpSessionHolder.getSession(customSessionId);
            if (session != null) {
                session.invalidate();
            }
        }
        // 获取默认的
        session = request.getSession(false);
        if (session != null) {
            session.invalidate();
        }

        SecurityContext context = SecurityContextHolder.getContext();
        SecurityContextHolder.clearContext();
        context.setAuthentication(null);
    }
}
