package com.elitescloud.boot.auth.provider.security.rememberme;

import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import lombok.Getter;
import lombok.Setter;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.rememberme.PersistentRememberMeToken;
import org.springframework.security.web.authentication.rememberme.PersistentTokenRepository;

import java.io.Serializable;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 记住我的信息存储.
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/13
 */
public class RedisRememberMeTokenRepository implements PersistentTokenRepository {
    private final String KEY_PREFIX = "cloudt:auth:rememberMe:";
    private final String URI_VALIDATE_REMEMBER_ME = "/oauth/login/attempt";

    private final RedisHelper redisHelper;
    private final int tokenTtlSeconds;

    public RedisRememberMeTokenRepository(RedisHelper redisHelper, int tokenTtlSeconds) {
        this.redisHelper = redisHelper;
        this.tokenTtlSeconds = tokenTtlSeconds;
    }

    @Override
    public void createNewToken(PersistentRememberMeToken token) {
        var authentication = SecurityContextHolder.getContext().getAuthentication();
        var request = HttpServletUtil.currentRequest();

        CustomPersistentRememberMeToken rememberMeToken = new CustomPersistentRememberMeToken();
        rememberMeToken.setUsername(token.getUsername());
        rememberMeToken.setSeries(token.getSeries());
        rememberMeToken.setTokenValue(token.getTokenValue());
        rememberMeToken.setDate(token.getDate());

        if (authentication instanceof AbstractCustomAuthenticationToken) {
            rememberMeToken.setLoginType(((AbstractCustomAuthenticationToken<?>) authentication).loginType().getType());
        }
        rememberMeToken.setPlatformId((Long) request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_PLATFORM_ID));

        redisHelper.execute(redisUtils -> {
            redisUtils.set(KEY_PREFIX + token.getSeries(), rememberMeToken, tokenTtlSeconds, TimeUnit.SECONDS);
            redisUtils.set(KEY_PREFIX + token.getUsername(), token.getSeries(), tokenTtlSeconds, TimeUnit.SECONDS);

            return null;
        });
    }

    @Override
    public void updateToken(String series, String tokenValue, Date lastUsed) {
        redisHelper.execute(redisUtils -> {
            CustomPersistentRememberMeToken rememberMeToken = (CustomPersistentRememberMeToken) redisUtils.get(KEY_PREFIX + series);
            rememberMeToken.setTokenValue(tokenValue);
            rememberMeToken.setDate(lastUsed);

            redisUtils.set(KEY_PREFIX + series, rememberMeToken, tokenTtlSeconds, TimeUnit.SECONDS);
            redisUtils.set(KEY_PREFIX + rememberMeToken.getUsername(), series, tokenTtlSeconds, TimeUnit.SECONDS);

            return null;
        });
    }

    @Override
    public PersistentRememberMeToken getTokenForSeries(String seriesId) {
        if (!URI_VALIDATE_REMEMBER_ME.equals(HttpServletUtil.currentRequest().getRequestURI())) {
            return null;
        }

        return null;

//        return redisHelper.execute(redisUtils -> {
//            CustomPersistentRememberMeToken rememberMeToken = (CustomPersistentRememberMeToken) redisUtils.get(KEY_PREFIX + seriesId);
//            if (rememberMeToken == null) {
//                return null;
//            }
//
//            var request = HttpServletUtil.currentRequest();
//            if (request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_PLATFORM_ID) == null) {
//                request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_PLATFORM_ID, rememberMeToken.getPlatformId());
//            }
//            if (request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_LOGIN_TYPE) == null) {
//                request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_LOGIN_TYPE, rememberMeToken.getLoginType());
//            }
//            request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_REMEMBER_ME_TOKEN, rememberMeToken);
//
//            return new PersistentRememberMeToken(rememberMeToken.getUsername(), rememberMeToken.getSeries(), rememberMeToken.getTokenValue(), rememberMeToken.getDate());
//        });
    }

    @Override
    public void removeUserTokens(String username) {
        redisHelper.execute(redisUtils -> {
            var series = (String) redisUtils.get(KEY_PREFIX + username);
            if (series == null) {
                return null;
            }

            redisUtils.del(KEY_PREFIX + series, KEY_PREFIX + username);

            return null;
        });
    }

    @Getter
    @Setter
    public static class CustomPersistentRememberMeToken implements Serializable {
        private static final long serialVersionUID = -5659593124002157202L;

        private String username;
        private String series;
        private String tokenValue;
        private Date date;

        private String loginType;
        private Long platformId;
    }
}
