package com.elitescloud.boot.auth.provider.sso2.support.convert;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.sso2.common.SsoConvertProperty;
import com.elitescloud.boot.auth.provider.sso2.common.SsoTypeEnum;
import com.elitescloud.boot.auth.provider.sso2.support.convert.properties.OidcSsoConvertProperty;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import org.jetbrains.annotations.Nullable;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.security.oauth2.core.AuthorizationGrantType;
import org.springframework.security.oauth2.core.endpoint.OAuth2ParameterNames;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.Map;

/**
 * oidc类型.
 *
 * @author Kaiser（wang shao）
 * @date 2025/6/8 周日
 */
public class OidcSsoAuthenticationConvert extends BasePlainSsoAuthenticationConvert {

    private final RestTemplateHelper restTemplateHelper;

    public OidcSsoAuthenticationConvert() {
        this.restTemplateHelper = RestTemplateHelper.instance(RestTemplateFactory.instance());
    }

    @Override
    public SsoTypeEnum supportType() {
        return SsoTypeEnum.OIDC;
    }

    @Override
    public <T extends SsoConvertProperty> Class<T> propertyType() {
        return (Class<T>) OidcSsoConvertProperty.class;
    }

    @Nullable
    @Override
    public <T extends SsoConvertProperty> InternalAuthenticationGranter.InternalAuthenticationToken convert(HttpServletRequest request, HttpServletResponse response, T properties) {
        OidcSsoConvertProperty props = (OidcSsoConvertProperty) properties;

        String value = getParam(request, props.getParamName(), props.getParamIn());
        if (CharSequenceUtil.isBlank(value)) {
            throw new IllegalArgumentException("参数为空:" + props.getParamName());
        }

        // 授权码换取token
        String accessToken = queryAccessToken(request, value, props);
        if (CharSequenceUtil.isBlank(accessToken)) {
            throw new BusinessException("查询认证服务器的授权token为空");
        }

        // 查询用户信息
        String username = queryUsername(accessToken, props);
        if (CharSequenceUtil.isBlank(username)) {
            throw new BusinessException("授权账户为空");
        }
        return new InternalAuthenticationGranter.InternalAuthenticationToken(props.getIdType(), username);
    }

    private String queryUsername(String accessToken, OidcSsoConvertProperty props) {
        MultiValueMap<String, String> headers = new LinkedMultiValueMap<>(4);
        headers.add(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken);

        Map<String, Object> userInfo = restTemplateHelper.exchange(props.getUserInfoEndpoint(), HttpMethod.GET, new HttpEntity<>(null, headers), new ParameterizedTypeReference<HashMap<String, Object>>() {});
        if (CollUtil.isEmpty(userInfo)) {
            return null;
        }
        return getValueByPath(props.getUserInfoParamPath(), userInfo);
    }

    private String queryAccessToken(HttpServletRequest request, String code, OidcSsoConvertProperty props) {
        MultiValueMap<String, Object> postParam = new LinkedMultiValueMap<>(8);
        postParam.add(OAuth2ParameterNames.CLIENT_ID, props.getAuthClientId());
        postParam.add(OAuth2ParameterNames.CLIENT_SECRET, props.getAuthClientSecret());
        postParam.add(OAuth2ParameterNames.GRANT_TYPE, AuthorizationGrantType.AUTHORIZATION_CODE.getValue());
        postParam.add(OAuth2ParameterNames.CODE, code);

        // 回调地址
        String redirectUri = getParam(request, props.getRedirectUriParam(), props.getParamIn());
        if (CharSequenceUtil.isBlank(redirectUri)) {
            redirectUri = props.getRedirectUri();
        }
        Assert.notBlank(redirectUri, "redirectUri为空");

        Map<String, Object> resp = restTemplateHelper.exchange(props.getAccessTokenEndpoint(), HttpMethod.POST, new HttpEntity<>(postParam), new ParameterizedTypeReference<HashMap<String, Object>>() {});
        if (CollUtil.isEmpty(resp)) {
            return null;
        }
        return (String) resp.get(OAuth2ParameterNames.ACCESS_TOKEN);
    }
}
