package com.elitescloud.boot.auth.util;

import com.elitescloud.boot.auth.model.Result;
import com.elitescloud.boot.auth.sso.common.SdkSsoConstants;
import com.elitescloud.boot.auth.sso.common.SdkSsoUriConstants;
import com.elitescloud.boot.auth.sso.model.UserInfoDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.Nullable;
import org.springframework.web.client.RestTemplate;

/**
 * 授权服务器交互工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/14
 */
public class AuthorizationServerHelper {
    private static final Logger LOG = LoggerFactory.getLogger(AuthorizationServerHelper.class);

    private final RestTemplate restTemplate;

    private AuthorizationServerHelper() {
        this.restTemplate = RestTemplateFactory.instance();
    }

    public static AuthorizationServerHelper getInstance() {
        return new AuthorizationServerHelper();
    }

    /**
     * ticket换取用户信息
     *
     * @param authServer 授权服务地址
     * @param ticket     ticket
     * @return 用户信息
     */
    @Nullable
    public UserInfoDTO ticket2UserInfo(String authServer, String ticket) {
        String url = normalizeUrl(authServer) + SdkSsoUriConstants.SSO_TICKET_TO_USER_INFO + "?" + SdkSsoConstants.PARAM_SSO_TICKET + "=" + ticket;
        try {
            ResponseEntity<Result<UserInfoDTO>> resp = restTemplate.exchange(url, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            });
            if (resp.getStatusCode().is2xxSuccessful() && resp.getBody() != null) {
                return resp.getBody().getData();
            }
            LOG.error("ticket换取用户信息结果：{}，{}", resp.getStatusCode(), resp.getBody());
            return null;
        } catch (Exception e) {
            LOG.error("请求路径：{}", url);
            LOG.error("ticket换取用户信息异常：", e);
            throw new IllegalStateException("请求认证服务器失败", e);
        }
    }

    /**
     * 登录调用授权服务生成ticket
     *
     * @param authServer  授权服务地址
     * @param account     账号
     * @param accountType 账号类型
     */
    public String generateTicket(String authServer, String account, String accountType) {
        String url = normalizeUrl(authServer) + SdkSsoUriConstants.SSO_TICKET_GENERATE +
                "?" + SdkSsoConstants.PARAM_SSO_ACCOUNT_TYPE + "=" + accountType
                + "&" + SdkSsoConstants.PARAM_SSO_ACCOUNT + "=" + account;
        try {
            ResponseEntity<Result<String>> resp = restTemplate.exchange(url, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            });
            if (resp.getStatusCode().is2xxSuccessful() && resp.getBody() != null) {
                return resp.getBody().getData();
            }
            LOG.info("ticket生成结果：{}，{}", resp.getStatusCode(), resp.getBody());
            return null;
        } catch (Exception e) {
            LOG.error("生成ticket路径：{}", url);
            LOG.error("生成ticket异常：", e);
            throw new IllegalStateException("生成ticket失败", e);
        }
    }

    /**
     * 注销登录ticket
     *
     * @param authServer 授权服务地址
     * @param ticket     ticket
     */
    public void revokeTicket(String authServer, String ticket) {
        String url = normalizeUrl(authServer) + SdkSsoUriConstants.SSO_TICKET_REVOKE + "?" + SdkSsoConstants.PARAM_SSO_TICKET + "=" + ticket;
        try {
            ResponseEntity<Result<String>> resp = restTemplate.exchange(url, HttpMethod.DELETE, null, new ParameterizedTypeReference<>() {
            });
            LOG.info("ticket注销结果：{}，{}", resp.getStatusCode(), resp.getBody());
        } catch (Exception e) {
            LOG.error("注销路径：{}", url);
            LOG.error("ticket注销异常：", e);
        }
    }

    /**
     * sso server调用各client进行注销
     *
     * @param clientAddress client地址
     * @param uri           注销uri
     * @param ticket        ticket
     */
    public void clientLogout(String clientAddress, String uri, String ticket) {
        String url = normalizeUrl(clientAddress) + uri + "?" + SdkSsoConstants.PARAM_SSO_TICKET + "=" + ticket;
        try {
            ResponseEntity<String> resp = restTemplate.exchange(url, HttpMethod.DELETE, null, new ParameterizedTypeReference<>() {
            });
            LOG.info("client注销结果：{}，{}", resp.getStatusCode(), resp.getBody());
        } catch (Exception e) {
            LOG.error("client注销路径：{}", url);
            LOG.error("client注销异常：", e);
        }
    }

    private String normalizeUrl(String url) {
        while (url.endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }

        return url;
    }
}
