package com.elitescloud.boot.auth.cas.provider;

import com.elitescloud.boot.auth.cas.task.ClientTokenHolder;
import com.elitescloud.boot.common.OpenApiException;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.NonNull;

import java.io.IOException;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
public abstract class BaseTransferHelper {
    private static final Logger LOG = LoggerFactory.getLogger(BaseTransferHelper.class);

    private final RestTemplateHelper restTemplateHelper;

    BaseTransferHelper(String authServer) {
        this.restTemplateHelper = RestTemplateHelper.instance(
                RestTemplateFactory.instance(builder -> builder.rootUri(authServer)
                        .additionalInterceptors(new AuthorizationInterceptor())
                ));
    }

    <T> T nullToDefault(T obj, T defaultObj) {
        return obj == null ? defaultObj : obj;
    }

    <T> T remoteExchange(String url, HttpMethod httpMethod, HttpEntity<?> httpEntity, ParameterizedTypeReference<T> responseType,
                         Object... param) {
        try {
            return restTemplateHelper.exchange(url, httpMethod, httpEntity, responseType, param);
        } catch (OpenApiException e) {
            if (e.getCode() == HttpStatus.UNAUTHORIZED.value()) {
                // 可能是token过期，发起重试
                LOG.info("重试请求：{}", url);
                ClientTokenHolder.refresh();
                return restTemplateHelper.exchange(url, httpMethod, httpEntity, responseType, param);
            }
            throw e;
        }
    }

    static class AuthorizationInterceptor implements ClientHttpRequestInterceptor {

        @NonNull
        @Override
        public ClientHttpResponse intercept(HttpRequest request, @NonNull byte[] body, @NonNull ClientHttpRequestExecution execution) throws IOException {
            HttpHeaders headers = request.getHeaders();
            var token = ClientTokenHolder.getToken();
            if (token == null) {
                throw new IllegalStateException("Token获取失败");
            }

            // 设置token
            headers.add(HttpHeaders.AUTHORIZATION, token.getTokenType() + " " + token.getAccessToken());
            return execution.execute(request, body);
        }
    }
}
