package com.elitescloud.boot.auth.cas.task;

import com.elitescloud.boot.auth.cas.UserSourceProvider;
import com.elitescloud.boot.auth.cas.provider.OAuth2ClientTemplate;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

/**
 * cas客户端相关任务调度.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/15
 */
public class CasClientScheduler implements SchedulingConfigurer {
    private static final Logger LOG = LoggerFactory.getLogger(CasClientScheduler.class);
    private final AuthorizationSdkProperties sdkProperties;
    private final TaskExecutor taskExecutor;

    private UserSourceProvider userSourceProvider;
    private OAuth2ClientTemplate oAuth2ClientTemplate;

    public CasClientScheduler(AuthorizationSdkProperties sdkProperties,
                              TaskExecutor taskExecutor) {
        this.sdkProperties = sdkProperties;
        this.taskExecutor = taskExecutor;
    }

    @Override
    public void configureTasks(@NonNull ScheduledTaskRegistrar taskRegistrar) {
        // 自动刷新客户端token
        addClientTokenRefreshTask(taskRegistrar);
        // 同步用户的任务
        addSyncUserTask(taskRegistrar);
    }

    private void addSyncUserTask(ScheduledTaskRegistrar taskRegistrar) {
        if (userSourceProvider == null) {
            return;
        }
        var casClient = sdkProperties.getCasClient();
        long interval = casClient.getSyncInterval() == null ? 0 : casClient.getSyncInterval().toMillis();
        if (interval < 1) {
            LOG.warn("CAS自动同步用户已关闭");
            return;
        }

        // 创建同步任务
        AutoSyncUserTask task = new AutoSyncUserTask(sdkProperties.getAuthServer(), userSourceProvider);
        taskRegistrar.addFixedDelayTask(() -> {
            if (!Boolean.TRUE.equals(sdkProperties.getCasClient().getEnabled())) {
                // 未启用
                return;
            }

            taskExecutor.execute(() -> {
                try {
                    task.setSyncSize(casClient.getSyncSize());
                    task.setRollbackAllOnException(Boolean.TRUE.equals(casClient.getRollBackAllOnException()));
                    task.run();
                } catch (Exception e) {
                    LOG.error("执行同步任务异常：", e);
                }
            });
        }, interval);
    }

    private void addClientTokenRefreshTask(ScheduledTaskRegistrar taskRegistrar) {
        if (oAuth2ClientTemplate == null) {
            return;
        }
        long interval = sdkProperties.getCasClient().getClientTokenInterval() == null ? 0 : sdkProperties.getCasClient().getClientTokenInterval().toMillis();
        if (interval < 1) {
            LOG.warn("CAS自动同步刷新客户端token已关闭");
            return;
        }
        LOG.info("CAS客户端刷新token间隔{}min", sdkProperties.getCasClient().getClientTokenInterval().toMinutes());

        AutoRefreshClientTokenTask task = new AutoRefreshClientTokenTask(oAuth2ClientTemplate);
        taskRegistrar.addFixedDelayTask(() -> {
            if (!Boolean.TRUE.equals(sdkProperties.getCasClient().getEnabled())) {
                // 未启用
                return;
            }

            taskExecutor.execute(() -> {
                try {
                    task.run();
                } catch (Exception e) {
                    LOG.error("执行刷新客户端token任务异常：", e);
                }
            });
        }, interval);
    }

    @Autowired(required = false)
    public void setUserSourceProvider(UserSourceProvider userSourceProvider) {
        this.userSourceProvider = userSourceProvider;
    }

    @Autowired(required = false)
    public void setoAuth2ClientTemplate(OAuth2ClientTemplate oAuth2ClientTemplate) {
        this.oAuth2ClientTemplate = oAuth2ClientTemplate;
    }
}
