package com.elitescloud.boot.auth.client.config;

import com.elitescloud.boot.auth.client.client.config.OAuthClientProperties;
import com.elitescloud.boot.auth.client.common.AuthorizationType;
import org.springframework.boot.context.properties.NestedConfigurationProperty;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 认证相关配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/31
 */
public class AuthorizationProperties {

    public static final String CONFIG_PREFIX = "elitesland.authorization";

    /**
     * 是否启用安全认证
     */
    private Boolean enabled = true;

    /**
     * 认证方式
     */
    private AuthorizationType type = AuthorizationType.SINGLE;

    /**
     * token生成时的有效期
     */
    private Duration tokenTtl = Duration.ofHours(2);

    /**
     * token自动续期时长，小于1时将不自动续期
     */
    private Duration tokenRenewal = Duration.ofHours(1);

    /**
     * token刷新频率，tokenRenewalRate * 2必须小于tokenRenewal
     */
    private Duration tokenRenewalRate = Duration.ofMillis(10);

    /**
     * 是否允许匿名访问
     */
    private Boolean anonymousEnabled = false;

    /**
     * 白名单，不允许匿名访问时有效
     */
    private Set<String> allowList = new HashSet<>(8);

    /**
     * 黑名单，允许匿名反问时有效
     */
    private Set<String> rejectList = new HashSet<>(8);

    /**
     * 是否启用csrf（跨站请求伪造），我们默认使用token，可避免，所以默认关闭即可
     */
    private Boolean csrfEnabled = false;

    /**
     * 是否启用cors
     */
    private Boolean corsEnabled = false;

    /**
     * 启用cors时有效的配置
     */
    private List<CorsConfig> cors = new ArrayList<>();

    /**
     * 是否启用session，使用token，所以默认自动判断
     */
    private Boolean sessionEnabled;

    /**
     * 客户端配置
     */
    @NestedConfigurationProperty
    private final OAuthClientProperties client = new OAuthClientProperties();

    /**
     * 角色前缀
     */
    private String rolePrefix = "ROLE_";

    /**
     * 登录页
     */
    private String loginPage = null;

    /**
     * 授权服务器地址
     */
    private String issuerUrl = null;

    /**
     * 重定向的路径前缀，nginx类代理导致重定向失败时的策略
     */
    private String redirectUriPrefix = null;

    public Boolean getEnabled() {
        return enabled;
    }

    public void setEnabled(Boolean enabled) {
        this.enabled = enabled;
    }

    public AuthorizationType getType() {
        return type;
    }

    public void setType(AuthorizationType type) {
        this.type = type;
    }

    public Duration getTokenTtl() {
        return tokenTtl;
    }

    public void setTokenTtl(Duration tokenTtl) {
        this.tokenTtl = tokenTtl;
    }

    public Duration getTokenRenewal() {
        return tokenRenewal;
    }

    public void setTokenRenewal(Duration tokenRenewal) {
        this.tokenRenewal = tokenRenewal;
    }

    public Duration getTokenRenewalRate() {
        return tokenRenewalRate;
    }

    public void setTokenRenewalRate(Duration tokenRenewalRate) {
        this.tokenRenewalRate = tokenRenewalRate;
    }

    public Boolean getAnonymousEnabled() {
        return anonymousEnabled;
    }

    public void setAnonymousEnabled(Boolean anonymousEnabled) {
        this.anonymousEnabled = anonymousEnabled;
    }

    public Set<String> getAllowList() {
        return allowList;
    }

    public void setAllowList(Set<String> allowList) {
        this.allowList = allowList;
    }

    public Set<String> getRejectList() {
        return rejectList;
    }

    public void setRejectList(Set<String> rejectList) {
        this.rejectList = rejectList;
    }

    public Boolean getCsrfEnabled() {
        return csrfEnabled;
    }

    public void setCsrfEnabled(Boolean csrfEnabled) {
        this.csrfEnabled = csrfEnabled;
    }

    public Boolean getCorsEnabled() {
        return corsEnabled;
    }

    public void setCorsEnabled(Boolean corsEnabled) {
        this.corsEnabled = corsEnabled;
    }

    public List<CorsConfig> getCors() {
        return cors;
    }

    public void setCors(List<CorsConfig> cors) {
        this.cors = cors;
    }

    public Boolean getSessionEnabled() {
        if (sessionEnabled != null) {
            return sessionEnabled;
        }
        // 默认oauth2认证时需要启用
        return AuthorizationType.OAUTH2_SERVER == getType();
    }

    public void setSessionEnabled(Boolean sessionEnabled) {
        this.sessionEnabled = sessionEnabled;
    }

    public OAuthClientProperties getClient() {
        return client;
    }

    public String getRolePrefix() {
        return rolePrefix;
    }

    public void setRolePrefix(String rolePrefix) {
        this.rolePrefix = rolePrefix;
    }

    public String getLoginPage() {
        return loginPage;
    }

    public void setLoginPage(String loginPage) {
        this.loginPage = loginPage;
    }

    public String getIssuerUrl() {
        return issuerUrl;
    }

    public void setIssuerUrl(String issuerUrl) {
        this.issuerUrl = issuerUrl;
    }

    public String getRedirectUriPrefix() {
        return redirectUriPrefix;
    }

    public void setRedirectUriPrefix(String redirectUriPrefix) {
        this.redirectUriPrefix = redirectUriPrefix;
    }

    public static class CorsConfig {
        /**
         * 匹配的路径，默认所有
         */
        private String pathMatcher = "/**";
        /**
         * 允许的源模式
         */
        private Set<String> allowedOriginPatterns = new HashSet<>();

        /**
         * 允许的源
         */
        private Set<String> allowedOrigins = new HashSet<>();

        /**
         * 允许的头
         */
        private Set<String> allowedHeaders = new HashSet<>();

        /**
         * 暴露的头
         */
        private Set<String> exposeHeaders = new HashSet<>();

        /**
         * 允许的方法
         */
        private Set<String> allowedMethods = new HashSet<>();

        /**
         * 是否信任的
         */
        private boolean allowCredentials;

        public String getPathMatcher() {
            return pathMatcher;
        }

        public void setPathMatcher(String pathMatcher) {
            this.pathMatcher = pathMatcher;
        }

        public Set<String> getAllowedOriginPatterns() {
            return allowedOriginPatterns;
        }

        public void setAllowedOriginPatterns(Set<String> allowedOriginPatterns) {
            this.allowedOriginPatterns = allowedOriginPatterns;
        }

        public Set<String> getAllowedOrigins() {
            return allowedOrigins;
        }

        public void setAllowedOrigins(Set<String> allowedOrigins) {
            this.allowedOrigins = allowedOrigins;
        }

        public Set<String> getAllowedHeaders() {
            return allowedHeaders;
        }

        public void setAllowedHeaders(Set<String> allowedHeaders) {
            this.allowedHeaders = allowedHeaders;
        }

        public Set<String> getExposeHeaders() {
            return exposeHeaders;
        }

        public void setExposeHeaders(Set<String> exposeHeaders) {
            this.exposeHeaders = exposeHeaders;
        }

        public Set<String> getAllowedMethods() {
            return allowedMethods;
        }

        public void setAllowedMethods(Set<String> allowedMethods) {
            this.allowedMethods = allowedMethods;
        }

        public boolean isAllowCredentials() {
            return allowCredentials;
        }

        public void setAllowCredentials(boolean allowCredentials) {
            this.allowCredentials = allowCredentials;
        }
    }
}
