package com.elitescloud.boot.auth.client.config.security.handler;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.util.ObjectMapperFactory;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.lang.NonNull;
import org.springframework.security.web.DefaultRedirectStrategy;
import org.springframework.security.web.RedirectStrategy;
import org.springframework.security.web.savedrequest.DefaultSavedRequest;
import org.springframework.security.web.savedrequest.SavedRequest;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/30
 */
@Log4j2
public abstract class AbstractHandler {

    protected final RedirectStrategy redirectStrategy = new DefaultRedirectStrategy();
    private ObjectMapper objectMapper;

    /**
     * 往response写返回内容
     *
     * @param result 返回对象
     */
    protected void writeResponse(@NonNull HttpServletResponse response, @NonNull Object result) throws IOException {
        writeResponse(response, result, HttpStatus.OK);
    }

    /**
     * 往response写返回内容
     *
     * @param response   response
     * @param result     返回对象
     * @param httpStatus 错误码，默认200
     */
    protected void writeResponse(@NonNull HttpServletResponse response, @NonNull Object result, HttpStatus httpStatus) throws IOException {
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setStatus(httpStatus == null ? HttpStatus.OK.value() : httpStatus.value());

        // 返回内容转字符串
        String content = null;
        try {
            content = result instanceof String ? (String) result : getObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(result);
        } catch (JsonProcessingException e) {
            log.error("返回response异常：", e);
            content = ObjectUtil.defaultIfNull(e.getCause(), e).getMessage();
        }

        // 写入返回内容
        try (var writer = response.getWriter()) {
            writer.write(content);
        } catch (IOException e) {
            log.error("返回结果失败：", e);
            throw e;
        }
    }

    /**
     * 重定向页面
     *
     * @param request
     * @param response
     * @param url
     */
    protected void sendRedirect(HttpServletRequest request, HttpServletResponse response, String url) throws IOException {
        // 允许跨域
        redirectStrategy.sendRedirect(request, response, url);
    }

    /**
     * 重定向
     * <p>
     * 用来解决由于nginx等代理导致的重定向时浏览器上路径出现错误
     *
     * @param uriPrefix
     * @param savedRequest
     * @param request
     * @param response
     */
    protected void sendRedirect(String uriPrefix, DefaultSavedRequest savedRequest, HttpServletRequest request, HttpServletResponse response) throws IOException {
        if (!StringUtils.hasText(uriPrefix)) {
            // 如果没有设置，则走默认逻辑
            sendRedirect(request, response, savedRequest.getRedirectUrl());
            return;
        }

        // 组装新的路径
        String url = this.obtainRedirectUrl(uriPrefix, savedRequest);
        sendRedirect(request, response, url);
    }

    /**
     * 获取重定向路径
     *
     * @param uriPrefix
     * @param savedRequest
     * @return
     */
    protected String obtainRedirectUrl(String uriPrefix, DefaultSavedRequest savedRequest) {
        if (!StringUtils.hasText(uriPrefix)) {
            return savedRequest.getRedirectUrl();
        }

        if (uriPrefix.endsWith("/")) {
            uriPrefix = uriPrefix.substring(0, uriPrefix.length() - 1);
        }
        String queryString = StringUtils.hasText(savedRequest.getQueryString()) ? "?" + savedRequest.getQueryString() : "";

        return uriPrefix + savedRequest.getRequestURI() + queryString;
    }

    /**
     * 是否支持重定向
     *
     * @param request
     * @return
     */
    protected boolean supportRedirect(HttpServletRequest request) {
        String redirect = request.getHeader(SecurityConstants.HEADER_AUTH_REDIRECT);
        return redirect == null || "true".equalsIgnoreCase(redirect);
    }

    /**
     * 是否支持重定向
     *
     * @param request
     * @return
     */
    protected boolean supportRedirect(SavedRequest request) {
        List<String> redirect = request.getHeaderValues(SecurityConstants.HEADER_AUTH_REDIRECT);
        if (CollectionUtils.isEmpty(redirect)) {
            return true;
        }

        for (String s : redirect) {
            if ("true".equalsIgnoreCase(s)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 打印参数
     *
     * @param request 请求
     */
    protected void printParameters(HttpServletRequest request) {
        var parametersQuery = HttpServletUtil.getQueryParameters(request);
        log.info("查询参数：{}", () -> {
            try {
                return getObjectMapper().writeValueAsString(parametersQuery);
            } catch (JsonProcessingException e) {
                log.error(request.getRequestURI() + "打印参数异常：", e);
            }
            return "";
        });
        var parametersForm = HttpServletUtil.getFormParameters(request);
        log.info("表单参数：{}", () -> {
            try {
                return getObjectMapper().writeValueAsString(parametersForm);
            } catch (JsonProcessingException e) {
                log.error(request.getRequestURI() + "打印参数异常：", e);
            }
            return "";
        });
    }

    /**
     * 打印请求头
     *
     * @param request request
     */
    protected void printHeaders(HttpServletRequest request) {
        var headers = HttpServletUtil.getHeaders(request);
        log.info("请求头：{}", () -> {
            try {
                return getObjectMapper().writeValueAsString(headers);
            } catch (JsonProcessingException e) {
                log.error(request.getRequestURI() + "打印请求头异常：", e);
            }
            return "";
        });
    }

    protected ObjectMapper getObjectMapper() {
        if (objectMapper == null) {
            try {
                objectMapper = SpringContextHolder.getBean(ObjectMapper.class);
            } catch (Exception e) {
                log.warn("获取ObjectMapper异常：{}", e.getMessage());
                objectMapper = ObjectMapperFactory.instance();
            }
        }

        return objectMapper;
    }
}
