package com.elitescloud.boot.auth.client.client.service;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.client.common.OAuthClientConstant;
import com.elitescloud.boot.auth.client.client.config.OAuthClientProperties;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.util.AuthorizationUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpHeaders;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.http.codec.json.Jackson2JsonDecoder;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;

import java.io.IOException;
import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/10/23
 */
@Log4j2
public class AuthServerRequestHolder {

    private final OAuthClientProperties oauthClientProperties;
    private final ObjectMapper objectMapper;

    private static final String AUTH_SERVER_METADATA_URI = "/.well-known/oauth-authorization-server";
    private Map<String, Object> authServerMetadata = Collections.emptyMap();

    private WebClient webClient;

    public AuthServerRequestHolder(OAuthClientProperties oauthClientProperties, String serverUri, ObjectMapper objectMapper) {
        this.oauthClientProperties = oauthClientProperties;
        this.objectMapper = objectMapper;
        init(serverUri);
    }

    public Mono<ApiResult<OAuthToken>> getAccessToken(String username, Terminal terminal) {
        var urlOptional = getTokenUriOfMetadata();

        MultiValueMap<String, Object> requestEntity = new LinkedMultiValueMap<>();
        requestEntity.add(OAuthClientConstant.SSO_PARAM_GRANT_TYPE, OAuthClientConstant.SSO_GRANT_TYPE);
        requestEntity.add(OAuthClientConstant.SSO_PARAM_USERNAME, username);
        requestEntity.add(OAuthClientConstant.SSO_PARAM_SCOPE, OAuthClientConstant.SCOPE_DEFAULT);
        requestEntity.add(OAuthClientConstant.SSO_PARAM_TERMINAL, terminal.name());

        String url = urlOptional.block();
        if (CharSequenceUtil.isBlank(url)) {
            return Mono.just(ApiResult.fail("授权服务异常，获取授权接口路径失败！"));
        }
        String clientAuth = null;
        try {
            clientAuth = AuthorizationUtil.encodeBasicAuth(OAuthClientConstant.CLIENT_ID, OAuthClientConstant.CLIENT_SECRET_DEFAULT);
        } catch (IOException e) {
            return Mono.just(ApiResult.fail("授权服务异常"));
        }

        return webClient.post()
                .uri(url)
                .header(HttpHeaders.AUTHORIZATION, clientAuth)
                .bodyValue(requestEntity)
                .retrieve()
                .bodyToMono(new ParameterizedTypeReference<ApiResult<OAuthToken>>() {
                })
                .doOnError(throwable -> log.error("调用授权服务器生成token异常：", throwable))
                .onErrorResume(Throwable.class, e -> Mono.just(ApiResult.fail("认证授权失败")))
                ;
    }

    private void init(String serverUri) {
        log.info("初始化OAuth服务认证客户端，服务器端地址：{}", Assert.notBlank(serverUri, "未知OAuth服务端地址"));

        createWebClient(serverUri);

        // 查询服务端元数据
        CompletableFuture.runAsync(() -> queryAuthServerMetadata().block());
    }

    private void createWebClient(String serverUri) {
        HttpClient httpClient = HttpClient.create()
                .baseUrl(serverUri)
                .responseTimeout(Objects.requireNonNullElse(oauthClientProperties.getReadTimeout(), Duration.ofSeconds(30)));
        ReactorClientHttpConnector httpConnector = new ReactorClientHttpConnector(httpClient);

        this.webClient = WebClient.builder()
                .clientConnector(httpConnector)
                .codecs(codec -> codec.defaultCodecs()
                        .jackson2JsonDecoder(new Jackson2JsonDecoder(objectMapper))
                )
                .build();
    }

    private Mono<Map<String, Object>> queryAuthServerMetadata() {
        return webClient.get()
                .uri(AUTH_SERVER_METADATA_URI)
                .retrieve()
                .bodyToMono(new ParameterizedTypeReference<Map<String, Object>>() {
                })
                .doOnError(throwable -> log.error("查询OAuth服务元数据失败：", throwable))
                .doOnNext(data -> this.authServerMetadata = data);
    }

    private Mono<String> getTokenUriOfMetadata() {
        String key = "token_endpoint";
        if (!authServerMetadata.isEmpty()) {
            String uri = (String) authServerMetadata.get(key);
            return uri == null ? Mono.empty() : Mono.just(uri);
        }
        return queryAuthServerMetadata()
                .filter(t -> !t.isEmpty() && t.containsKey(key))
                .map(t -> (String) t.get(key));
    }
}
