package com.elitescloud.boot.auth.client.client.service;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.elitescloud.boot.auth.client.client.OAuthUserProvider;
import com.elitescloud.boot.auth.client.client.config.OAuthClientProperties;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.CacheControl;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/10/23
 */
@Log4j2
public class OAuthClientUserService {

    private final OAuthClientProperties oauthClientProperties;
    private final OAuthUserProvider userProvider;
    private final ObjectMapper objectMapper;
    private final AuthServerRequestHolder authServerRequestHolder;

    private static final String PARAM_CLIENT = "_c";

    public OAuthClientUserService(OAuthClientProperties oauthClientProperties, OAuthUserProvider userService,
                                  ObjectMapper objectMapper, AuthServerRequestHolder authServerRequestHolder) {
        this.oauthClientProperties = oauthClientProperties;
        this.userProvider = userService;
        this.objectMapper = objectMapper;
        this.authServerRequestHolder = authServerRequestHolder;
    }

    public void grantToken(HttpServletRequest request, HttpServletResponse response) throws IOException {
        var reqParam = ServletUtil.getParamMap(request);
        log.info("Oauth Client Token请求：" + objectMapper.writeValueAsString(reqParam));

        boolean authenticated;
        try {
            authenticated = userProvider.authenticate(reqParam);
        } catch (Exception e) {
            if (e instanceof BusinessException) {
                log.info("Oauth Client认证不通过：{}", e.getMessage());

                writeBusinessException(response, e);
                return;
            }
            log.error("Oauth Client认证异常：", e);
            writeResponse(response, ApiResult.fail(ApiCode.SYSTEM_EXCEPTION));
            return;
        }

        if (!authenticated) {
            // 未认证通过
            writeResponse(response, ApiResult.fail("认证不通过，请确认用户状态是否正常"));
            return;
        }

        // 获取用户信息
        String username = null;
        try {
            username = loadUser();
        } catch (Exception e) {
            if (e instanceof BusinessException) {
                log.info("获取认证用户信息异常：{}", e.getMessage());
                writeBusinessException(response, e);
                return;
            }

            log.error("获取认证用户信息异常：", e);
            writeResponse(response, ApiResult.fail("认证不通过，请确认用户状态是否正常"));
            return;
        }
        if (CharSequenceUtil.isBlank(username)) {
            writeResponse(response, ApiResult.fail("颁发token失败，未查询到有效用户信息"));
            return;
        }

        // 给用户生成token
        generateToken(response, reqParam, username);
    }

    private void writeBusinessException(HttpServletResponse response, Exception e) throws IOException {
        BusinessException eb = (BusinessException) e;
        var result = ApiResult.fail(eb.getCode() == null ? ApiCode.SYSTEM_EXCEPTION : eb.getApiCode(), eb.getMessage());
        writeResponse(response, result);
    }

    private void generateToken(HttpServletResponse response, Map<String, String> reqParam, String username) throws IOException {
        log.info("为用户【{}】生成token", username);

        var tokenResult = authServerRequestHolder.getAccessToken(username, userProvider.terminal()).block();
        writeResponse(response, tokenResult);
    }

    private String loadUser() {
        var user = userProvider.obtainUser();
        if (user != null) {
            return user;
        }

        return oauthClientProperties.getSpecifiedUser();
    }

    private void writeResponse(HttpServletResponse response, ApiResult<?> result) throws IOException {
        response.setCharacterEncoding(StandardCharsets.UTF_8.toString());
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setHeader(HttpHeaders.CACHE_CONTROL, CacheControl.noCache().getHeaderValue());
        response.setHeader(HttpHeaders.PRAGMA, CacheControl.noStore().getHeaderValue());

        var writer = response.getWriter();
        var prettyPrinter = objectMapper.writerWithDefaultPrettyPrinter();
        try {
            writer.write(prettyPrinter.writeValueAsString(result));
        } catch (JsonProcessingException e) {
            writer.write(prettyPrinter.writeValueAsString(ApiResult.fail(ApiCode.FAIL)));
        }
        writer.flush();
        writer.close();
    }
}
