package com.elitescloud.boot.auth.client.config.security.resolver.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.auth.client.config.security.resolver.BearerTokenResolver;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.lang.NonNull;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * bearerToken解析器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/2
 */
@Slf4j
public class DefaultBearerTokenResolver implements BearerTokenResolver {
    private static final Pattern AUTHORIZATION_PATTERN = Pattern.compile("^Bearer (?<token>[a-zA-Z0-9-._~+/]+=*)$",
            Pattern.CASE_INSENSITIVE);
    private static final String TOKEN_PREFIX = SecurityConstants.HEADER_TOKEN_PREFIX;

    private boolean allowFormEncodedBodyParameter = false;

    private boolean allowUriQueryParameter = false;

    private String bearerTokenHeaderName = HttpHeaders.AUTHORIZATION;

    @Override
    public String resolve(HttpServletRequest request) throws AuthenticationException {
        // 先从请求头解析
        String authorizationHeaderToken = resolveFromAuthorizationHeader(request);
        if (StringUtils.hasText(authorizationHeaderToken)) {
            return authorizationHeaderToken;
        }

        if (isParameterTokenSupportedForRequest(request) && isParameterTokenEnabledForRequest(request)) {
            return resolveFromRequestParameters(request);
        }

        return null;
    }

    public void setAllowFormEncodedBodyParameter(boolean allowFormEncodedBodyParameter) {
        this.allowFormEncodedBodyParameter = allowFormEncodedBodyParameter;
    }

    public void setAllowUriQueryParameter(boolean allowUriQueryParameter) {
        this.allowUriQueryParameter = allowUriQueryParameter;
    }

    public void setBearerTokenHeaderName(@NonNull String bearerTokenHeaderName) {
        this.bearerTokenHeaderName = bearerTokenHeaderName;
    }

    private String resolveFromAuthorizationHeader(HttpServletRequest request) {
        String authorization = CharSequenceUtil.blankToDefault(request.getHeader(this.bearerTokenHeaderName), request.getHeader(SecurityConstants.HEADER_ACCESS_TOKEN));
        if (!StringUtils.startsWithIgnoreCase(authorization, TOKEN_PREFIX)) {
            return null;
        }
        Matcher matcher = AUTHORIZATION_PATTERN.matcher(authorization);
        if (!matcher.matches()) {
            log.info("不支持的bearer token：{}", authorization);
            return null;
        }
        return matcher.group("token");
    }

    private static String resolveFromRequestParameters(HttpServletRequest request) {
        String[] values = request.getParameterValues(SecurityConstants.REQUEST_PARAM_TOKEN);
        if (values == null || values.length == 0) {
            return null;
        }
        if (values.length == 1) {
            return values[0];
        }
        throw new AuthorizationException("token格式不正确");
    }

    private boolean isParameterTokenSupportedForRequest(final HttpServletRequest request) {
        return (("POST".equalsIgnoreCase(request.getMethod())
                && MediaType.APPLICATION_FORM_URLENCODED_VALUE.equalsIgnoreCase(request.getContentType()))
                || "GET".equalsIgnoreCase(request.getMethod()));
    }

    private boolean isParameterTokenEnabledForRequest(final HttpServletRequest request) {
        return ((this.allowFormEncodedBodyParameter && "POST".equalsIgnoreCase(request.getMethod())
                && MediaType.APPLICATION_FORM_URLENCODED_VALUE.equalsIgnoreCase(request.getContentType()))
                || (this.allowUriQueryParameter && "GET".equalsIgnoreCase(request.getMethod())));
    }
}
