package com.elitescloud.boot.auth.client.config.support.dubbo;

import com.elitescloud.boot.auth.client.config.support.AuthenticationContext;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.condition.ConditionalOnRpc;
import com.elitescloud.boot.constant.DubboConstant;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.rpc.RpcContext;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.util.StringUtils;

/**
 * Dubbo相关Security配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/3
 */
@Import({SecurityDubboConfig.AuthorizationOnDubboConfig.class, SecurityDubboConfig.AuthorizationOnNoDubboConfig.class})
@ConditionalOnRpc(requiredDubbo = true)
public class SecurityDubboConfig {

    /**
     * 当有dubbo时的配置
     */
    @Log4j2
    static class AuthorizationOnDubboConfig {

        @Bean
        public AuthenticationContext dubboAuthenticationContext() {
            return new AuthenticationContext() {
                @Override
                public Long getUserId() {
                    String userIdStr = getParamFromDubbo(DubboConstant.CURRENT_USER_ID);
                    if (StringUtils.hasText(userIdStr)) {
                        return Long.parseLong(userIdStr);
                    }
                    return null;
                }

                @Override
                public String getUsername() {
                    String username = getParamFromDubbo(DubboConstant.CURRENT_USER_NAME);
                    if (StringUtils.hasText(username)) {
                        return username;
                    }

                    return null;
                }

                @Override
                public Long getTenantId() {
                    String tenantIdStr = getParamFromDubbo(DubboConstant.CURRENT_TENANT_ID);
                    if (StringUtils.hasText(tenantIdStr)) {
                        return Long.parseLong(tenantIdStr);
                    }

                    return null;
                }

                @Override
                public GeneralUserDetails getUser() {
                    String token = getParamFromDubbo(DubboConstant.CURRENT_AUTH_TOKEN);
                    if (!StringUtils.hasText(token)) {
                        // 没有传递用户token
                        log.info("dubbo未传递用户token");
                        return null;
                    }

                    // 解析token
                    return SecurityContextUtil.convertToken(token);
                }

                /**
                 * 从dubbo上下文获取参数
                 *
                 * @param key 参数名称
                 * @return 参数值
                 */
                private String getParamFromDubbo(String key) {
                    return RpcContext.getServiceContext().getAttachment(key);
                }
            };
        }
    }

    /**
     * 当没有dubbo时的配置
     */
    @ConditionalOnMissingClass("org.apache.dubbo.config.bootstrap.DubboBootstrap")
    static class AuthorizationOnNoDubboConfig {
    }
}
