package com.elitescloud.boot.auth.client.config;

import com.elitescloud.boot.auth.client.config.security.AbstractServletSecurityConfig;
import com.elitescloud.boot.auth.client.config.security.OAuth2ResourceServletSecurityConfig;
import com.elitescloud.boot.auth.client.config.security.SingleClientServletSecurityConfig;
import com.elitescloud.boot.auth.client.config.support.springcloud.SecuritySpringCloudConfig;
import com.elitescloud.boot.threadpool.common.ContextTransfer;
import lombok.extern.log4j.Log4j2;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.factory.PasswordEncoderFactories;
import org.springframework.security.crypto.password.DelegatingPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.SecurityFilterChain;

/**
 * 认证中心客户端自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/20
 */
@SuppressWarnings({"deprecation", "removal"})
@EnableConfigurationProperties(com.elitescloud.cloudt.authorization.api.client.config.AuthorizationProperties.class)
@Import({AuthorizationClientAutoConfiguration.OnEnableSecurity.class, AuthorizationClientAutoConfiguration.OnDisableSecurity.class,})
@Log4j2
public class AuthorizationClientAutoConfiguration {

    @Bean
    public PasswordEncoder passwordEncoder() {
        DelegatingPasswordEncoder encoder = (DelegatingPasswordEncoder) PasswordEncoderFactories.createDelegatingPasswordEncoder();
        encoder.setDefaultPasswordEncoderForMatches(new BCryptPasswordEncoder());
        return encoder;
    }

    /**
     * 启用安全配置
     */
    @ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
    @Import({SingleClientServletSecurityConfig.class, OAuth2ResourceServletSecurityConfig.class,
            SecuritySpringCloudConfig.class})
    static class OnEnableSecurity {
        private final AuthorizationProperties authorizationProperties;

        public OnEnableSecurity(AuthorizationProperties authorizationProperties) {
            this.authorizationProperties = authorizationProperties;
            log.info("启用Security");
        }

        /**
         * 用户认证信息上下文传递器
         *
         * @return 上下文传递器
         */
        @Bean
        ContextTransfer<SecurityContext> contextTransferSecurityContext() {
            return new ContextTransfer<>() {
                @Override
                public SecurityContext getContext() {
                    return SecurityContextHolder.getContext();
                }

                @Override
                public void setContext(SecurityContext context) {
                    SecurityContextHolder.setContext(context);
                }

                @Override
                public void clearContext() {
                    SecurityContextHolder.clearContext();
                }
            };
        }
    }


    /**
     * 禁用安全配置
     */
    @ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "enabled", havingValue = "false")
    static class OnDisableSecurity {
        public OnDisableSecurity() {
            log.info("禁用Security");
        }

        @Bean
        @ConditionalOnMissingBean(name = AbstractServletSecurityConfig.SECURITY_CHAIN_DEFAULT)
        @Order(Ordered.HIGHEST_PRECEDENCE)
        public SecurityFilterChain defaultFilterChain(HttpSecurity http) throws Exception {
            http.authorizeRequests().anyRequest().permitAll();

            return http.build();
        }
    }
}
