package com.elitescloud.boot.auth.client.config;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.CloudtSecurityDefaultAutoConfiguration;
import com.elitescloud.boot.auth.AuthorizedClient;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.client.config.support.RedisAuthenticationCache;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.provider.CurrentUserProvider;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.boot.redis.CloudtRedisAutoConfiguration;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;

import javax.validation.constraints.NotBlank;

/**
 * 认证缓存自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/15
 */
@AutoConfigureAfter(CloudtRedisAutoConfiguration.class)
@AutoConfigureBefore(CloudtSecurityDefaultAutoConfiguration.class)
@ConditionalOnBean(RedisUtils.class)
public class CloudtAuthorizationCacheAutoConfiguration {

    private final RedisUtils redisUtils;

    public CloudtAuthorizationCacheAutoConfiguration(RedisUtils redisUtils) {
        this.redisUtils = redisUtils;
    }

    /**
     * redis操作工具
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    RedisHelper redisHelper(ObjectProvider<RedisWrapper> wrapperObjectProvider) {
        return new RedisHelper(redisUtils, wrapperObjectProvider.getIfAvailable());
    }

    /**
     * 身份缓存服务
     *
     * @param redisHelper
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    AuthenticationCache defaultAuthenticationCache(RedisHelper redisHelper) {
        return new RedisAuthenticationCache(redisHelper);
    }

    /**
     * 用户信息
     *
     * @return
     */
    @Bean
    UserDetailProvider userDetailProvider() {
        return new UserDetailProvider() {
            @Override
            public GeneralUserDetails getByToken(@NotBlank String token) {
                if (CharSequenceUtil.isBlank(token)) {
                    return null;
                }

                // 去掉token类型
                var tokenArray = token.split(" ");
                token = tokenArray.length == 1 ? tokenArray[0] : tokenArray[1];

                return SecurityContextUtil.convertToken(token);
            }

            @Override
            public GeneralUserDetails currentUser() {
                return SecurityContextUtil.currentUser();
            }
        };
    }

    @Bean
    CurrentUserProvider currentUserProvider() {
        return new CurrentUserProvider() {
            @Override
            public SysUserDTO currentUser() {
                var user = SecurityContextUtil.currentUser();
                return user == null ? null : user.getUser();
            }

            @Override
            public AuthorizedClient currentClient() {
                return SecurityContextUtil.currentAuthorizedClient();
            }
        };
    }
}
