package com.elitescloud.boot.auth.client.config.security.configurer;

import com.elitescloud.boot.auth.client.common.UserValidator;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.client.config.security.configurer.filter.AccessTokenRenewalFilter;
import com.elitescloud.boot.auth.client.config.security.configurer.filter.BearerTokenAuthenticationFilter;
import com.elitescloud.boot.auth.client.config.security.configurer.provider.BearerTokenAuthenticationProvider;
import com.elitescloud.boot.auth.client.config.security.handler.DefaultAuthenticationEntryPointHandler;
import com.elitescloud.boot.auth.client.config.security.resolver.BearerTokenResolver;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.config.annotation.web.HttpSecurityBuilder;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.access.intercept.AuthorizationFilter;
import org.springframework.security.web.authentication.AnonymousAuthenticationFilter;
import org.springframework.security.web.context.SecurityContextRepository;
import org.springframework.util.Assert;

/**
 * 认证客户端相关security扩展配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/6
 */
@Log4j2
public class DefaultAuthorizationConfigurer<H extends HttpSecurityBuilder<H>>
        extends AbstractHttpConfigurer<DefaultAuthorizationConfigurer<H>, H> {

    private final AuthorizationProperties authorizationProperties;
    private final AuthenticationCache authenticationCache;
    private final JwtDecoder jwtDecoder;
    private final ObjectProvider<AuthorizationConfigurerCustomizer> authorizationConfigurerCustomizer;

    public DefaultAuthorizationConfigurer(AuthorizationProperties authorizationProperties,
                                          AuthenticationCache authenticationCache,
                                          JwtDecoder jwtDecoder,
                                          ObjectProvider<AuthorizationConfigurerCustomizer> authorizationConfigurerCustomizer) {
        this.authorizationProperties = authorizationProperties;
        this.authenticationCache = authenticationCache;
        this.jwtDecoder = jwtDecoder;
        this.authorizationConfigurerCustomizer = authorizationConfigurerCustomizer;
    }

    /**
     * 未认证时的处理
     */
    private AuthenticationEntryPoint authenticationEntryPoint;
    /**
     * 是否引入bearerToken Filter
     */
    private boolean needBearerTokenAuthenticationFilter = false;

    /**
     * 是否自动续期token
     */
    private boolean autoRenewalToken = false;

    /**
     * token解析器
     */
    private BearerTokenResolver bearerTokenResolver;

    /**
     * 用户校验器
     */
    private UserValidator userValidator;

    @Override
    public void init(H builder) throws Exception {
        Assert.notNull(authorizationProperties, "AuthorizationProperties为空");
        builder.setSharedObject(AuthorizationProperties.class, authorizationProperties);

        // 自定义处理
        if (authorizationConfigurerCustomizer != null) {
            for (AuthorizationConfigurerCustomizer customizer : authorizationConfigurerCustomizer) {
                customizer.init(builder);
            }
        }

        super.init(builder);
    }

    @Override
    public void configure(H builder) throws Exception {
        // 添加从header中识别token的filter
        addBearerTokenAuthenticationFilter(builder);

        // 自动刷新token Filter
        if (autoRenewalToken) {
            addAccessTokenRenewalFilter(builder);
        }

        // 自定义处理
        if (authorizationConfigurerCustomizer != null) {
            for (AuthorizationConfigurerCustomizer customizer : authorizationConfigurerCustomizer) {
                customizer.configure(builder);
            }
        }

        super.configure(builder);
    }

    /**
     * 是否需要加入bearerToken认证filter
     *
     * @param needBearerTokenAuthenticationFilter
     * @return
     */
    public DefaultAuthorizationConfigurer<H> needBearerTokenAuthenticationFilter(boolean needBearerTokenAuthenticationFilter) {
        this.needBearerTokenAuthenticationFilter = needBearerTokenAuthenticationFilter;
        return this;
    }

    /**
     * 是否支持自动续期token
     *
     * @param autoRenewalToken
     * @return
     */
    public DefaultAuthorizationConfigurer<H> autoRenewalToken(boolean autoRenewalToken) {
        this.autoRenewalToken = autoRenewalToken;
        return this;
    }

    /**
     * 未认证时的处理
     *
     * @param authenticationEntryPoint
     * @return
     */
    public DefaultAuthorizationConfigurer<H> authenticationEntryPoint(AuthenticationEntryPoint authenticationEntryPoint) {
        this.authenticationEntryPoint = authenticationEntryPoint;
        return this;
    }

    /**
     * token解析器
     *
     * @param bearerTokenResolver
     * @return
     */
    public DefaultAuthorizationConfigurer<H> bearerTokenResolver(BearerTokenResolver bearerTokenResolver) {
        this.bearerTokenResolver = bearerTokenResolver;
        return this;
    }

    /**
     * 用户解析器
     *
     * @param userValidator
     * @return
     */
    public DefaultAuthorizationConfigurer<H> userValidator(UserValidator userValidator) {
        this.userValidator = userValidator;
        return this;
    }

    private void addAccessTokenRenewalFilter(H http) {
        if (authenticationCache == null) {
            // 不支持用户信息缓存
            return;
        }

        var filter = new AccessTokenRenewalFilter(authorizationProperties, authenticationCache, jwtDecoder);
        if (bearerTokenResolver != null) {
            filter.setBearerTokenResolver(bearerTokenResolver);
        }
        if (userValidator != null) {
            filter.setUserValidator(userValidator);
        }
        http.addFilterAfter(filter, AuthorizationFilter.class);
    }

    private void addBearerTokenAuthenticationFilter(H http) {
        // 判断是否已引入oauth2 resource，如果已引入，则不添加
        try {
            Class.forName("org.springframework.security.oauth2.server.resource.BearerTokenAuthenticationToken");
            log.debug("已存在BearerTokenAuthenticationFilter");
            if (!needBearerTokenAuthenticationFilter) {
                return;
            }
        } catch (Exception ignored) {
        }

        // 添加filter
        var authenticationManager = http.getSharedObject(AuthenticationManager.class);
        BearerTokenAuthenticationFilter filter = new BearerTokenAuthenticationFilter(authenticationManager);
        if (bearerTokenResolver != null) {
            filter.setBearerTokenResolver(bearerTokenResolver);
        }

        if (authenticationEntryPoint == null) {
            authenticationEntryPoint = new DefaultAuthenticationEntryPointHandler(authorizationProperties.getLoginPage());
        }
        filter.setAuthenticationEntryPoint(authenticationEntryPoint);
        http.addFilterBefore(filter, AnonymousAuthenticationFilter.class);

        var securityContextRepository = http.getSharedObject(SecurityContextRepository.class);
        if (securityContextRepository != null) {
            filter.setSecurityContextRepository(securityContextRepository);
        }

        // 添加provider
        BearerTokenAuthenticationProvider authenticationProvider = new BearerTokenAuthenticationProvider();
        http.authenticationProvider(postProcess(authenticationProvider));
    }
}
