package com.elitescloud.boot.auth.client.config.security.configurer.provider;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.AuthorizedClient;
import com.elitescloud.boot.auth.BearerTokenAuthenticationToken;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.common.InterceptUri;
import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.constant.AuthenticationClaim;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.jwt.BadJwtException;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.web.servlet.util.matcher.MvcRequestMatcher;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.web.servlet.handler.HandlerMappingIntrospector;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.*;

/**
 * bearerToken身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/2
 */
@Log4j2
public class BearerTokenAuthenticationProvider implements AuthenticationProvider {
    private static final Duration DEFAULT_MAX_CLOCK_SKEW = Duration.of(5, ChronoUnit.MINUTES);
    private static final Cache<String, Authentication> AUTHENTICATION_CACHE = Caffeine.newBuilder()
            .maximumSize(1000)
            .expireAfterWrite(Duration.ofMinutes(5))
            .build();

    private final Clock clock = Clock.systemUTC();
    private final List<RequestMatcher> allowRequestList = new ArrayList<>();

    private AuthenticationCache authenticationCache;
    private JwtDecoder jwtDecoder;
    private AuthorizationProperties authorizationProperties;
    private HandlerMappingIntrospector handlerMappingIntrospector;

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        BearerTokenAuthenticationToken authenticationToken = (BearerTokenAuthenticationToken) authentication;
        Jwt jwt = this.decodeToken(authenticationToken);
        if (jwt == null) {
            if (allow()) {
                // 允许访问
                return SecurityConstants.AUTHENTICATION_ANONYMOUS;
            }
            throw new AuthorizationException("当前用户还未认证或身份认证已过期");
        }

        return obtainJwtAuthentication(jwt);
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return BearerTokenAuthenticationToken.class.isAssignableFrom(authentication);
    }

    private Authentication obtainJwtAuthentication(Jwt jwt) {
        String principalType = jwt.getClaimAsString(AuthenticationClaim.KEY_PRINCIPAL_TYPE);

        var authentication = AUTHENTICATION_CACHE.getIfPresent(jwt.getTokenValue());
        if (authentication != null) {
            if (CharSequenceUtil.equals(principalType, AuthenticationClaim.VALUE_PRINCIPAL_USER)) {
                boolean valid = authenticationCache.exists(jwt.getTokenValue());
                // token已失效
                if (!valid) {
                    if (allow()) {
                        // 允许访问
                        return SecurityConstants.AUTHENTICATION_ANONYMOUS;
                    }
                    log.info("无效token：{}", jwt.getTokenValue());
                    throw new AuthorizationException("当前用户还未认证或身份认证已过期");
                }
            }
            return authentication;
        }


        // 认证成功
        BearerTokenAuthenticationToken authorizationResult = new BearerTokenAuthenticationToken(jwt.getTokenValue(), Collections.emptyList());

        authorizationResult.setAuthorizedClient(AuthorizedClient.buildByJwt(jwt));

        if (CharSequenceUtil.equals(principalType, AuthenticationClaim.VALUE_PRINCIPAL_USER)) {
            // 如果是用户认证，则获取用户信息
            GeneralUserDetails user = authenticationCache.getUserDetail(jwt.getTokenValue());
            if (user == null) {
                if (allow()) {
                    // 允许访问
                    return SecurityConstants.AUTHENTICATION_ANONYMOUS;
                }
                log.info("无效token：{}", jwt.getTokenValue());
                throw new AuthorizationException("当前用户还未认证或身份认证已过期");
            }

            authorizationResult.setPrincipal(user);
        } else {
            // 校验token是否有效
            validateExpires(jwt);
        }

        AUTHENTICATION_CACHE.put(jwt.getTokenValue(), authorizationResult);
        return authorizationResult;
    }

    private boolean allow() {
        if (authorizationProperties.getAnonymousEnabled()) {
            // 允许匿名访问
            return true;
        }

        // 是否在白名单内
        if (allowRequestList.isEmpty()) {
            // 初始化白名单
            initAllowRequestList();
        }
        if (allowRequestList.isEmpty()) {
            return false;
        }

        var request = HttpServletUtil.currentRequest();
        for (RequestMatcher requestMatcher : allowRequestList) {
            if (requestMatcher.matches(request)) {
                return true;
            }
        }

        return false;
    }

    private void initAllowRequestList() {
        Set<String> allowList = new HashSet<>();
        // 配置的白名单
        if (authorizationProperties.getAllowList() != null) {
            allowList.addAll(authorizationProperties.getAllowList());
        }
        // 内置白名单
        allowList.addAll(InterceptUri.getAllowUri());
        if (!allowList.isEmpty()) {
            HandlerMappingIntrospector introspector = ObjectUtil.defaultIfNull(handlerMappingIntrospector, new HandlerMappingIntrospector());
            for (String s : allowList) {
                if (adapterMvcRequestMatch(s)) {
                    allowRequestList.add(new MvcRequestMatcher(introspector, s));
                } else {
                    allowRequestList.add(new AntPathRequestMatcher(s));
                }
            }
        }
    }

    /**
     * 是否适配mvc匹配模式
     *
     * @param pattern
     * @return
     */
    private boolean adapterMvcRequestMatch(String pattern) {
        if (pattern == null) {
            return false;
        }
        // mvc匹配模式下，**只能位于结尾
        var indexDoubleWildcard = pattern.indexOf("**");
        return indexDoubleWildcard == pattern.length() - 2;
    }

    private Jwt decodeToken(BearerTokenAuthenticationToken bearer) {
        try {
            return jwtDecoder.decode(bearer.getToken());
        } catch (BadJwtException e) {
            log.warn("解析token异常, 不支持的token:{}, ", bearer.getToken(), e);
//            throw new InvalidBearerTokenException("不支持的token", e);
        }
        return null;
    }

    private void validateExpires(Jwt jwt) {
        Instant expiry = jwt.getExpiresAt();
        if (expiry != null) {
            if (Instant.now(this.clock).minus(DEFAULT_MAX_CLOCK_SKEW).isAfter(expiry)) {
                throw new AuthorizationException("身份认证已过期");
            }
        }
        Instant notBefore = jwt.getNotBefore();
        if (notBefore != null) {
            if (Instant.now(this.clock).plus(DEFAULT_MAX_CLOCK_SKEW).isBefore(notBefore)) {
                throw new AuthorizationException("身份认证还未生效");
            }
        }
    }

    private boolean renewal() {
        return authorizationProperties.getTokenRenewal() != null && authorizationProperties.getTokenRenewal().getSeconds() > 0;
    }

    @Autowired
    public void setAuthenticationCache(AuthenticationCache authenticationCache) {
        this.authenticationCache = authenticationCache;
    }

    @Autowired
    public void setJwtDecoder(JwtDecoder jwtDecoder) {
        this.jwtDecoder = jwtDecoder;
    }

    @Autowired
    public void setAuthorizationProperties(AuthorizationProperties authorizationProperties) {
        this.authorizationProperties = authorizationProperties;
    }

    @Autowired(required = false)
    @Lazy
    public void setHandlerMappingIntrospector(HandlerMappingIntrospector handlerMappingIntrospector) {
        this.handlerMappingIntrospector = handlerMappingIntrospector;
    }
}
