package com.elitescloud.boot.auth.client.config.support;

import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

/**
 * redis身份认证缓存.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/2
 */
@Log4j2
public class RedisAuthenticationCache implements AuthenticationCache {

    private final RedisHelper redisHelper;

    public RedisAuthenticationCache(RedisHelper redisHelper) {
        this.redisHelper = redisHelper;
    }

    @Override
    public void setUserDetail(@NonNull String key, @NonNull GeneralUserDetails user, Duration timeToLive) {
        try {
            redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateCacheKey(key);
                if (timeToLive != null) {
                    redisUtils.set(cacheKey, user, timeToLive.toSeconds());
                    return null;
                }

                redisUtils.set(cacheKey, user);
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("缓存用户详细信息失败", e);
        }
    }

    @Override
    public GeneralUserDetails getUserDetail(@NonNull String key) {
        try {
            return redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateCacheKey(key);
                return (GeneralUserDetails) redisUtils.get(cacheKey);
            });
        } catch (Exception e) {
            throw new BusinessException("获取用户详细信息失败", e);
        }
    }

    @Override
    public void removeUserDetail(@NonNull String key) {
        try {
            redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateCacheKey(key);
                redisUtils.del(cacheKey);

                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("删除用户详细信息失败", e);
        }
    }

    @Override
    public void expireAt(@NonNull String key, Duration ttl) {
        try {
            redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateCacheKey(key);
                if (ttl == null) {
                    redisUtils.del(cacheKey);
                    return null;
                }

                redisUtils.expire(cacheKey, ttl.toSeconds(), TimeUnit.SECONDS);
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("设置用户过期时间失败", e);
        }
    }

    @Override
    public boolean exists(@NonNull String key) {
        try {
            return redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateCacheKey(key);
                return redisUtils.hasKey(cacheKey);
            });
        } catch (Exception e) {
            throw new BusinessException("判断用户缓存是否存在失败", e);
        }
    }

    @Override
    public void setAttribute(@NonNull String key, @NonNull Object attribute, Duration timeToLive) {
        try {
            redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateAttributeCacheKey(key);
                if (timeToLive != null) {
                    redisUtils.set(cacheKey, attribute, timeToLive.toSeconds());
                    return null;
                }

                redisUtils.set(cacheKey, attribute);
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("缓存用户扩展属性失败", e);
        }
    }

    @Override
    public Object getAttribute(@NonNull String key) {
        try {
            return redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateAttributeCacheKey(key);
                return redisUtils.get(cacheKey);
            });
        } catch (Exception e) {
            throw new BusinessException("获取用户属性失败", e);
        }
    }

    @Override
    public void removeAttribute(@NonNull String key) {
        try {
            redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateAttributeCacheKey(key);
                redisUtils.del(cacheKey);

                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("删除用户属性信息失败", e);
        }
    }

    @Override
    public void expireAttributeAt(@NonNull String key, Duration ttl) {
        try {
            redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateAttributeCacheKey(key);
                if (ttl == null) {
                    redisUtils.del(cacheKey);
                    return null;
                }

                redisUtils.expire(cacheKey, ttl.toSeconds(), TimeUnit.SECONDS);
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("设置用户过期时间失败", e);
        }
    }

    @Override
    public boolean existsAttribute(@NonNull String key) {
        try {
            return redisHelper.execute(redisUtils -> {
                var cacheKey = this.generateAttributeCacheKey(key);
                return redisUtils.hasKey(cacheKey);
            });
        } catch (Exception e) {
            throw new BusinessException("判断用户缓存是否存在失败", e);
        }
    }

    private String generateCacheKey(String cacheKey) {
        return SecurityConstants.CACHE_PREFIX_CURRENT_USER + cacheKey;
    }

    private String generateAttributeCacheKey(String cacheKey) {
        return SecurityConstants.CACHE_PREFIX_CURRENT_USER_ATTRIBUTE + cacheKey;
    }
}
