package com.elitescloud.boot.datasecurity.dpr.service.impl;

import cn.zhxu.bs.util.MapBuilder;
import cn.zhxu.bs.util.MapUtils;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.auth.util.SecurityUtil;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.service.RoleDataPermissionRuleCacheInterface;
import com.elitescloud.boot.datasecurity.dpr.service.RoleDataPermissionRuleService;
import com.elitescloud.boot.datasecurity.dpr.service.util.AnnotationUtil;
import com.elitescloud.boot.datasecurity.dpr.service.util.DataPermissionRuleServiceUtil;
import com.elitescloud.boot.datasecurity.dpr.service.util.RoleDatePermissionColumnUtil;
import com.elitescloud.boot.datasecurity.dpr.service.util.RoleDatePermissionRowUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.system.dto.SysDpcRoleApiFieldsDTO;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据权限规则解析接口  需要bean Searcher和jpa分别实现。
 *
 * @author : chen
 * @date 2022-11-24 14:12
 */
@Slf4j
public class RoleDataPermissionRuleServiceImpl implements RoleDataPermissionRuleService {
    @Autowired
    private TenantClientProvider tenantClientProvider;

    private final RoleDataPermissionRuleCacheInterface roleDataPermissionRuleCacheInterface;

    public RoleDataPermissionRuleServiceImpl(RoleDataPermissionRuleCacheInterface roleDataPermissionRuleCacheInterface) {
        this.roleDataPermissionRuleCacheInterface = roleDataPermissionRuleCacheInterface;
    }

    @Override
    public SysDprRoleApiRowColumnRuleDTO getUserRoleDpr() {
        var token = SecurityContextUtil.currentToken();
        if (token == null || "".equals(token)) {
            log.error("数据权限：获取Token为空 ");
            SecurityUtil.throwUnauthorizedException();
            return null;
        }

        var rule = roleDataPermissionRuleCacheInterface.getTokenDprLocalCache(token)
                .orElseGet(() -> roleDataPermissionRuleCacheInterface.getTokenDprRedisCache(token)
                        .orElseGet(() -> roleDataPermissionRuleCacheInterface.roleDataPermissionRuleRpc(token)));
        return rule;
    }

    @Override
    public String getBeanSearcherGroupExpr(Map<String, Object> params) {

        return RoleDatePermissionColumnUtil.getBeanSearcherGroupExpr(params);
    }

    @Override
    public void addSetBeanSearcherGroupExpr(Map<String, Object> params, String groupExpr) {
        RoleDatePermissionColumnUtil.addSetBeanSearcherGroupExpr(params, groupExpr);
    }

    @Override
    public MapBuilder tenantAuthSearcherHandleEngine(Class<?> classBean, Map<String, Object> params, String... pathObj) {
        var roelDprDto = getUserRoleDpr();
        StringBuilder groupAuthExpr = new StringBuilder();

        MapBuilder mapBuilder = MapUtils.builder(params);
        //设置租户条件，并返回分组表达式
        StringBuilder groupTenantExpr = new StringBuilder(
                RoleDatePermissionColumnUtil.tenantAuthBuilder(tenantClientProvider, mapBuilder, params));

        //行数据权限配置 根据请求的path路径  比对当前用户全部的api行数据权限配置。 找到对应的api数据权限配置规则组和规则信息并返回
        List<SysDprRoleApiDataRuleListQueryDTO> startDto =
                RoleDatePermissionColumnUtil.apiPathRoleApiRowsRuleGroupDtoHandle(pathObj, roelDprDto);
        var fields = AnnotationUtil.getAllFields(classBean);
        if (!startDto.isEmpty()) {

            for (SysDprRoleApiDataRuleListQueryDTO sdto : startDto) {
                boolean ifField = false;
                for (Field field : fields) {
                    if (field.getName().equals(sdto.getDprRuleField())) {
                        ifField = true;
                        break;
                    }
                }
                if (!ifField) {
                    throw new BusinessException("异常：对应的规则字段在bean中不存在-" + sdto.getDprRuleField());
                }
            }

            //设置规则条件并返回拼接的分组表达式
            StringBuilder ruleGroupExpr = ruleAuthBuilder(mapBuilder, startDto);
            //拼接分组表达式
            if (ruleGroupExpr.length() > 0) {
                groupAuthExpr = groupTenantExpr.append("&").append(ruleGroupExpr);
            }
        } else {
            groupAuthExpr = groupTenantExpr;
        }
        //如果至少根默认SQL条件 + 租户条件 是不需要设置分组表达式的 直接根和租户条件and关系拼接  只有增加了数据规则权限 才生效表达式。
        //如果没有数据权限规则，只是设置了默认+租户，设置了表达式。 会造成租户条件不生效。
        if (groupAuthExpr.length() > 0) {
            //设置分组条件表达式
            mapBuilder.groupExpr(groupAuthExpr.toString());
        } else {
            throw new RuntimeException("数据权限分组表达式 groupAuthExpr is null  ");
        }

        //列数据权限设置
        List<SysDpcRoleApiFieldsDTO> columnDto =
                RoleDatePermissionRowUtil.apiPathRoleApiColumnRuleGroupDtoHandle(pathObj, roelDprDto);
        if (columnDto != null && columnDto.size() > 0) {
            //设置列权限字段  多个角色的同一个API的 展示类型合并成一个。
            RoleDatePermissionColumnUtil.setColumnMapBuilder(fields, columnDto, mapBuilder);
        }

        return mapBuilder;
    }


    /**
     * 根据规则构造行数据权限条件
     **/
    @Override
    public StringBuilder ruleAuthBuilder(MapBuilder mapBuilder,
                                         List<SysDprRoleApiDataRuleListQueryDTO> ruleList) {

        StringBuilder groupExpr = new StringBuilder();
        //分组
        Map<Long, List<SysDprRoleApiDataRuleListQueryDTO>> groupBy =
                ruleList.stream().collect(
                        Collectors.groupingBy(SysDprRoleApiDataRuleListQueryDTO::getRoleId));
        int groupStrIndex = 0;
        int groupMapIndex = 0;

        for (Long key : groupBy.keySet()) {
            var v = groupBy.get(key);
            //排序
            Collections.sort(v, Comparator.comparing(SysDprRoleApiDataRuleListQueryDTO::getRuleOrder));
            for (int i = 0; i < v.size(); i++) {
                var rule = v.get(i);
                //beanSearcher的分组关系设置。由于同一个字段，不同条件的or
                String gStr = RoleDatePermissionColumnUtil.GROUP_STR[groupStrIndex++];
                //设置分组
                mapBuilder.group(gStr);
                //拼接BS的分组条件 groupExpr直接拼接
                RoleDatePermissionColumnUtil.jointGroupExpr(i, v.size(), rule, gStr, groupExpr);
                String valueType = rule.getDprRuleValueType();
                //目前规则值都已经在第一次远程调用接口的时候计算好了。所以各类型的判断暂时无用。都采用同一个方法处理。
                //固定值类型值设置  //自定义类型值设置  和固定值一样 //系统内置
                if (
//                        valueType.equals(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_FIXED.name()) ||
                        valueType.equals(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM.name())
                                || valueType.equals(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name())
                                || valueType.equals(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name())) {
                    DataPermissionRuleServiceUtil.dprRuleValueTypeMapBuilder(mapBuilder, rule);
                } else {
                    DataPermissionRuleServiceUtil.throwRuntimeException("非法的value-type类型，请尽快联系管理员。：", rule);
                }
            }
            if (++groupMapIndex < groupBy.size()) {
                groupExpr.append("|");
            }

        }
        groupExpr.insert(0, "(");
        groupExpr.append(")");
        return groupExpr;
    }

    @Override
    public MapBuilder tenantSearcherHandleEngine(Class<?> beanClass, Map<String, Object> paraMap, String... pathObj) {
        //根据请求的path路径  比对当前用户全部的api数据权限配置。 找到对应的api数据权限配置规则组和规则信息并返回
        MapBuilder mapBuilder = MapUtils.builder(paraMap);
        //根据原SQL的条件分组关系+租户条件分组关系表达式，构建一个新的分组关系字符串给规则分组表达式使用
        StringBuilder groupExpr = new StringBuilder(
                RoleDatePermissionColumnUtil.tenantBuilder(tenantClientProvider, mapBuilder, paraMap));

        //如果至少根默认SQL条件 + 租户条件 是不需要设置分组表达式的 直接根和租户条件and关系拼接  只有增加了数据规则权限 才生效表达式。
        //如果没有数据权限规则，只是设置了默认+租户，设置了表达式。 会造成租户条件不生效。
        return mapBuilder;
    }


}
