package com.elitescloud.boot.datasecurity.common;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.datasecurity.common.extension.FieldPermissionExtension;
import com.elitescloud.boot.datasecurity.config.DataSecurityProperties;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleRelationEnum;
import com.elitescloud.boot.datasecurity.dpr.service.RoleDataPermissionRuleService;
import com.elitescloud.boot.datasecurity.jpa.strategy.RuleStrategyManager;
import com.elitescloud.boot.util.ArrayUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.system.dto.SysDpcRoleApiFieldsDTO;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import com.querydsl.core.BooleanBuilder;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.PathBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.PropertyAccessorFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * 数据权限工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/4/17
 */
@Slf4j
public class DataSecurityUtil {
    private static RoleDataPermissionRuleService roleDataPermissionRuleService;
    private static DataSecurityProperties dataSecurityProperties;
    private static List<FieldPermissionExtension> fieldPermissionExtensions;

    private static final InheritableThreadLocal<BusinessContext> BUSINESS_CONTEXT = new InheritableThreadLocal<>();

    private DataSecurityUtil() {
    }

    /**
     * 生成数据权限的过滤条件
     * <p>
     * 自动匹配当前用户和当前请求
     *
     * @param entityClass 实体类
     * @return 条件
     */
    public static Predicate predicateForJPA(@NotNull Class<?> entityClass) {
        return predicateForJPA(entityClass, null);
    }

    /**
     * 生成数据权限的过滤条件
     * <p>
     * 自动匹配当前用户和当前请求
     *
     * @param entityClass   实体类
     * @param alias         实体类的别名
     * @param ruleFieldName 权限字段
     * @return 条件
     */
    public static Predicate predicateForJPA(@NotNull Class<?> entityClass, String alias, String... ruleFieldName) {
        Assert.notNull(entityClass, "实体类为空");
        if (CharSequenceUtil.isBlank(alias)) {
            var entityName = entityClass.getSimpleName();
            alias = entityName.substring(0, 1).toLowerCase() + entityName.substring(1);
        }
        try {
            var userRoleDpr = queryDataPermission();
            return getAuthJpaPredicate(userRoleDpr, entityClass, alias, ruleFieldName);
        } catch (Exception e) {
            log.error("数据权限条件生成异常：{}", entityClass.getName(), e);
            return new BooleanBuilder(null);
        }
    }

    /**
     * 数据权限清洗数据
     * <p>
     * 抹去不可读的字段
     *
     * @param data 待清洗数据
     */
    public static void eraseByFieldPermission(Object data) {
        if (ObjUtil.isEmpty(data)) {
            return;
        }

        // 获取不可读字段
        var fieldNames = getUnreadableFields();
        if (fieldNames.isEmpty()) {
            return;
        }

        // 修改属性值
        var start = System.currentTimeMillis();
        ObjUtil.unwrap(data, d -> {
            Set<String> eraseFieldNames = new HashSet<>(fieldNames);
            // 获取自定义扩展中要清洗的字段
            eraseFieldNames.addAll(getUnreadableFieldsByExtension(d, Collections.unmodifiableSet(fieldNames)));

            // 根据字段清洗数据对象
            eraseObjectValue(d, eraseFieldNames, getDataSecurityProperties().getFieldPermission().getPlaceholder());
        });
        if (log.isErrorEnabled()) {
            log.debug("erase cost {}ms  by field permission", System.currentTimeMillis() - start);
        }
    }

    /**
     * 在指定业务对象下执行
     *
     * @param businessObjectCode    业务对象编码
     * @param businessOperationCode 业务对象的操作编码
     * @param executor              待执行的方法
     * @param <T>                   结果类型
     * @return 待执行的方法返回的结果
     */
    public static <T> T executeByTargetBusiness(String businessObjectCode, String businessOperationCode, @NotNull Supplier<T> executor) {
        var parentContext = BUSINESS_CONTEXT.get();
        // 设置新的业务对象为上下文
        BUSINESS_CONTEXT.set(new BusinessContext(businessObjectCode, businessOperationCode));
        try {
            return executor.get();
        } finally {
            // 上下文还原
            if (parentContext == null) {
                BUSINESS_CONTEXT.remove();
            } else {
                BUSINESS_CONTEXT.set(parentContext);
            }
        }
    }

    /**
     * 获取当前用户的数据权限
     * <p>
     * 匹配当前求的权限规则
     *
     * @return 数据权限
     */
    public static SysDprRoleApiRowColumnRuleDTO getDataPermission() {
        return queryDataPermission();
    }

    /**
     * 获取用户的所有数据权限规则
     *
     * @return 数据权限
     */
    public static SysDprRoleApiRowColumnRuleDTO getAllDataPermission() {
        return getRoleDataPermissionRuleService().getOriginalDataPermission();
    }

    /**
     * 获取当前用户的字段权限
     * <p>
     * 匹配当前求的权限规则
     *
     * @return 字段权限
     */
    public static List<SysDpcRoleApiFieldsDTO> getFieldPermission() {
        // 判断是否启用
        var prop = getDataSecurityProperties();
        if (Boolean.FALSE.equals(prop.getEnabled()) || Boolean.FALSE.equals(prop.getFieldPermission().getEnabled())) {
            log.info("field permission is disabled");
            return Collections.emptyList();
        }

        // 获取权限规则
        var dataPermission = queryDataPermission();
        if (dataPermission == null || CollUtil.isEmpty(dataPermission.getSysDpcRoleApiFieldsDTOList())) {
            return Collections.emptyList();
        }

        // 过滤出字段
        boolean conflictShow = Boolean.TRUE.equals(getDataSecurityProperties().getConflictShow());
        return new ArrayList<>(dataPermission.getSysDpcRoleApiFieldsDTOList().stream()
                .collect(Collectors.toMap(SysDpcRoleApiFieldsDTO::getFieldName, Function.identity(), (t1, t2) -> {
                    if (conflictShow) {
                        // 冲突时有权限优先
                        if (Boolean.TRUE.equals(t1.getReadable())) {
                            if (Boolean.TRUE.equals(t2.getReadable())) {
                                return Boolean.TRUE.equals(t1.getWriteable()) ? t1 : t2;
                            }
                            return t1;
                        }
                        return Boolean.TRUE.equals(t2.getReadable()) ? t2 : t1;
                    }

                    // 无权限优先
                    if (Boolean.FALSE.equals(t1.getReadable())) {
                        if (Boolean.FALSE.equals(t2.getReadable())) {
                            return Boolean.FALSE.equals(t1.getWriteable()) ? t1 : t2;
                        }
                        return t1;
                    }
                    return Boolean.FALSE.equals(t2.getReadable()) ? t2 : t1;
                })).values());
    }

    private static SysDprRoleApiRowColumnRuleDTO queryDataPermission() {
        String businessObjectCode = null, businessOperationCode = null;
        // 先判断是否有上下文
        var businessContext = BUSINESS_CONTEXT.get();
        if (businessContext != null) {
            businessObjectCode = businessContext.businessObjectCode;
            businessOperationCode = businessContext.businessOperationCode;
        }

        var service = getRoleDataPermissionRuleService();
        if (StringUtils.hasText(businessObjectCode) || StringUtils.hasText(businessOperationCode)) {
            // 指定了业务对象
            var dataPermission = service.getOriginalDataPermission();
            if (dataPermission == null) {
                return null;
            }
            return service.matchDataPermission(dataPermission, businessObjectCode, businessOperationCode);
        }

        // 默认根据当前请求
        return service.getMatchedDataPermission();
    }

    private static void eraseObjectValue(@NotNull Object data, @NotEmpty Set<String> eraseFieldNames, String placeholder) {
        // 清洗数据
        var propertyAccessor = PropertyAccessorFactory.forDirectFieldAccess(data);
        Class<?> fieldType = null;
        Object value = null;
        for (String fieldName : eraseFieldNames) {
            if (CharSequenceUtil.isBlank(fieldName)) {
                continue;
            }
            try {
                fieldType = propertyAccessor.getPropertyType(fieldName);
                if (fieldType == null) {
                    log.warn("{}.{}不存在，将忽略", data.getClass().getName(), fieldName);
                    continue;
                }
                // 转换抹去后的值
                value = eraseValue(fieldType, placeholder);
                propertyAccessor.setPropertyValue(fieldName, value);
            } catch (BeansException e) {
                log.warn("数据权限清洗{}.{}失败：", data.getClass().getName(), fieldName, e);
            }
        }
    }

    private static Set<String> getUnreadableFieldsByExtension(Object bean, Set<String> fieldNames) {
        Set<String> result = new HashSet<>();

        Set<String> tempFieldNames = null;
        for (var extension : getFieldPermissionExtensions()) {
            try {
                tempFieldNames = extension.getUnreadableFields(bean, fieldNames);
            } catch (Exception e) {
                log.error("{}获取不可读字段出现异常：", extension.getClass().getName(), e);
                continue;
            }

            if (CollUtil.isNotEmpty(tempFieldNames)) {
                result.addAll(tempFieldNames);
            }
        }

        return result;
    }

    private static Set<String> getUnreadableFields() {
        var fields = getFieldPermission();
        if (fields.isEmpty()) {
            return Collections.emptySet();
        }

        // 过滤出需要忽略掉的字段
        return fields.stream()
                .filter(t -> Boolean.FALSE.equals(t.getReadable()))
                .map(SysDpcRoleApiFieldsDTO::getFieldName)
                .collect(Collectors.toSet());
    }

    private static Object eraseValue(@NotNull Class<?> type, String placeholder) {
        if (String.class.isAssignableFrom(type)) {
            // 字符串的支持配置的替换符
            return CharSequenceUtil.blankToDefault(placeholder, null);
        }

        if (!type.isPrimitive()) {
            // 非基本类型
            return null;
        }

        // 基本类型
        if (type.equals(byte.class)) {
            return 0;
        }
        if (type.equals(short.class)) {
            return 0;
        }
        if (type.equals(int.class)) {
            return 0;
        }
        if (type.equals(long.class)) {
            return 0;
        }
        if (type.equals(float.class)) {
            return 0;
        }
        if (type.equals(double.class)) {
            return 0;
        }
        if (type.equals(char.class)) {
            return 0;
        }
        if (type.equals(boolean.class)) {
            return false;
        }

        return null;
    }

    /***
     * 返回当前用户的访问的API的菜单的角色权限的JPA Predicate条件
     * 如果菜单编码为空将会抛异常，强制要求前端传递过来。
     * @param entityClass jpa实体类
     * @param alias jpa sql查询表的别名，如果没有其就是jpa实体类的首字母小写名称 如SysUser 应该是sysUser
     * @param ruleFieldName 可以null，需要指定规则字段配料的字段字符串返回规则。  特殊情况使用，如：只需要生成某个字段规则的条件，其他的不需要。
     * **/
    private static Predicate getAuthJpaPredicate(SysDprRoleApiRowColumnRuleDTO userRoleDpr, Class<?> entityClass, String alias, String[] ruleFieldName) {
        if (userRoleDpr == null || CollUtil.isEmpty(userRoleDpr.getSysDprRoleApiDataRuleListQueryDTO())) {
            log.info("没有找到当前用户数据权限配置-跳过数据权限");
            return new BooleanBuilder(null);
        }
        // 行数据权限配置 根据请求的path路径  比对当前用户全部的api行数据权限配置。 找到对应的api数据权限配置规则组和规则信息并返回
        List<SysDprRoleApiDataRuleListQueryDTO> startDto = userRoleDpr.getSysDprRoleApiDataRuleListQueryDTO();
        if (ArrayUtil.isNotEmpty(ruleFieldName)) {
            var ruleFieldNameList = Arrays.asList(ruleFieldName);
            startDto = startDto.stream()
                    .filter(dpr -> ruleFieldNameList.contains(dpr.getDprRuleField()))
                    .collect(Collectors.toList());
        }
        if (CollUtil.isEmpty(startDto)) {
            log.info("没有找到当前用户数据权限配置匹配规则-跳过数据权限");
            return new BooleanBuilder(null);
        }

        // queryDsl动态构造
        PathBuilder<?> entityPath = new PathBuilder<Object>(entityClass, alias);
        BooleanBuilder returnPredicate = new BooleanBuilder();
        List<Predicate> groupPredicates = new ArrayList<Predicate>();

        var groupMap = startDto.stream().collect(Collectors.groupingBy(SysDprRoleApiDataRuleListQueryDTO::getRoleCode));
        //根据角色分组，每个角色直接的条件采用(（角色1条件）or（角色2条件）)方式
        for (var entry : groupMap.entrySet()) {
            BooleanBuilder groupPredicate = new BooleanBuilder();
            for (var sysRule : entry.getValue()) {
                try {
                    var predicateRule = RuleStrategyManager.getPathAutomatically(entityPath, sysRule);
                    //单个角色内的or或and条件
                    if (DprRuleRelationEnum.DPR_RULE_RELATION_AND.name().equals(sysRule.getDprRuleRelation())) {
                        groupPredicate.and(predicateRule);
                    } else if (DprRuleRelationEnum.DPR_RULE_RELATION_OR.name().equals(sysRule.getDprRuleRelation())) {
                        groupPredicate.or(predicateRule);
                    } else {
                        log.error("规则的关系异常:{}", sysRule);
                    }

                } catch (Exception e) {
                    log.error("规则字段匹配异常：", e);
                }
            }
            if (groupPredicate.hasValue()) {
                //不同角色的先存在List中
                groupPredicates.add(groupPredicate.getValue());
            }

        }
        Predicate[] integersArray2 = groupPredicates.toArray(Predicate[]::new);
        return returnPredicate.andAnyOf(integersArray2);
    }

    private static RoleDataPermissionRuleService getRoleDataPermissionRuleService() {
        if (roleDataPermissionRuleService == null) {
            roleDataPermissionRuleService = SpringContextHolder.getBean(RoleDataPermissionRuleService.class);
        }
        return roleDataPermissionRuleService;
    }

    private static DataSecurityProperties getDataSecurityProperties() {
        if (dataSecurityProperties == null) {
            dataSecurityProperties = SpringContextHolder.getBean(DataSecurityProperties.class);
        }
        return dataSecurityProperties;
    }

    private static List<FieldPermissionExtension> getFieldPermissionExtensions() {
        if (fieldPermissionExtensions == null) {
            fieldPermissionExtensions = SpringContextHolder.getObjectProvider(FieldPermissionExtension.class).stream().collect(Collectors.toList());
        }
        return fieldPermissionExtensions;
    }

    private static class BusinessContext {
        private final String businessObjectCode;
        private final String businessOperationCode;

        public BusinessContext(String businessObjectCode, String businessOperationCode) {
            this.businessObjectCode = businessObjectCode;
            this.businessOperationCode = businessOperationCode;
        }
    }
}
