package com.elitescloud.boot.datasecurity.dpr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.auth.util.SecurityUtil;
import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.constant.CompatibleModeEnum;
import com.elitescloud.boot.constant.OpenFeignConstant;
import com.elitescloud.boot.constant.WebConstant;
import com.elitescloud.boot.datasecurity.config.DataSecurityProperties;
import com.elitescloud.boot.datasecurity.dpr.service.RoleDataPermissionRuleCacheService;
import com.elitescloud.boot.datasecurity.dpr.service.RoleDataPermissionRuleService;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.system.constant.DataPermissionType;
import com.elitescloud.cloudt.system.dto.BaseDataSecurityRuleDTO;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.NotNull;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 数据权限规则解析接口  需要bean Searcher和jpa分别实现。
 *
 * @author : chen
 * @date 2022-11-24 14:12
 */
@Slf4j
public class RoleDataPermissionRuleServiceImpl implements RoleDataPermissionRuleService {

    private final RoleDataPermissionRuleCacheService roleDataPermissionRuleCacheInterface;
    private final DataSecurityProperties properties;

    public RoleDataPermissionRuleServiceImpl(RoleDataPermissionRuleCacheService roleDataPermissionRuleCacheInterface,
                                             DataSecurityProperties properties) {
        this.roleDataPermissionRuleCacheInterface = roleDataPermissionRuleCacheInterface;
        this.properties = properties;
    }

    @Override
    public SysDprRoleApiRowColumnRuleDTO getMatchedDataPermission() {
        var request = HttpServletUtil.currentRequest();
        if (request == null) {
            return null;
        }

        // 查询数据权限
        var dataPermission = this.queryDataPermission(request);

        // 获取业务对象信息
        var businessObjectCode = this.obtainBusinessObjectCode(request);
        var businessOperationCode = this.obtainBusinessOperationCode(request);
        if (StringUtils.hasText(businessOperationCode) && CharSequenceUtil.isBlank(businessObjectCode)) {
            businessObjectCode = this.queryBusinessObjectCodeByBusinessOperationCode(businessOperationCode);
        }
        // 根据业务对象信息匹配规则
        return this.matchDataPermission(dataPermission, businessObjectCode, businessOperationCode);
    }

    @Override
    public SysDprRoleApiRowColumnRuleDTO getOriginalDataPermission() {
        return this.queryDataPermission(HttpServletUtil.currentRequest());
    }

    @Override
    public SysDprRoleApiRowColumnRuleDTO matchDataPermission(SysDprRoleApiRowColumnRuleDTO ruleDTO, String businessObjectCode, String businessOperationCode) {
        var request = HttpServletUtil.currentRequest();
        if (ruleDTO == null || request == null) {
            return null;
        }
        if (CollUtil.isEmpty(ruleDTO.getSysDprRoleApiDataRuleListQueryDTO()) && CollUtil.isEmpty(ruleDTO.getSysDpcRoleApiFieldsDTOList())) {
            return ruleDTO;
        }

        // 获取业务对象编码
        if (StringUtils.hasText(businessObjectCode)) {
            businessObjectCode = businessObjectCode.contains(":") ? businessObjectCode.split(":")[0] : businessObjectCode;
        } else if (StringUtils.hasText(businessOperationCode)) {
            businessObjectCode = this.queryBusinessObjectCodeByBusinessOperationCode(businessOperationCode);
        }

        // 匹配当前请求的规则
        log.info("start to match row rule...");
        ruleDTO.setSysDprRoleApiDataRuleListQueryDTO(this.filterRule(request, businessObjectCode, businessOperationCode, ruleDTO.getSysDprRoleApiDataRuleListQueryDTO()));
        log.info("start to match field rule...");
        ruleDTO.setSysDpcRoleApiFieldsDTOList(Boolean.FALSE.equals(properties.getFieldPermission().getEnabled()) ?
                Collections.emptyList() : this.filterRule(request, businessObjectCode, businessOperationCode, ruleDTO.getSysDpcRoleApiFieldsDTOList()));

        return ruleDTO;
    }

    private SysDprRoleApiRowColumnRuleDTO queryDataPermission(HttpServletRequest request) {
        // 判断是否支持数据权限
        if (!this.supportDataPermission(request)) {
            log.info("do not support data permission");
            return null;
        }

        // 查询权限规则
        var token = SecurityContextUtil.currentToken();
        if (!StringUtils.hasText(token)) {
            log.error("数据权限：获取Token为空 ");
            SecurityUtil.throwUnauthorizedException();
            return null;
        }
        var rule = roleDataPermissionRuleCacheInterface.getTokenDprLocalCache(token)
                .orElseGet(() -> roleDataPermissionRuleCacheInterface.getTokenDprRedisCache(token)
                        .orElseGet(() -> roleDataPermissionRuleCacheInterface.roleDataPermissionRuleRpc(token)));
        if (rule == null) {
            return null;
        }

        // 过滤规则
        SysDprRoleApiRowColumnRuleDTO filteredRule = new SysDprRoleApiRowColumnRuleDTO();
        filteredRule.setUserId(rule.getUserId());
        filteredRule.setTenantId(rule.getTenantId());
        filteredRule.setTenantOrgId(rule.getTenantOrgId());
        filteredRule.setRoelIdList(rule.getRoelIdList());
        filteredRule.setRoleCodeList(rule.getRoleCodeList());
        filteredRule.setSysDprRoleApiDataRuleListQueryDTO(rule.getSysDprRoleApiDataRuleListQueryDTO());
        filteredRule.setSysDpcRoleApiFieldsDTOList(Boolean.FALSE.equals(properties.getFieldPermission().getEnabled()) ?
                Collections.emptyList() : rule.getSysDpcRoleApiFieldsDTOList());

        return filteredRule;
    }

    private boolean supportDataPermission(HttpServletRequest request) {
        if (request == null) {
            // 非web请求
            log.info("not a web request");
            return false;
        }
        if (Boolean.FALSE.equals(properties.getEnabled())) {
            // 已禁用数据权限
            return false;
        }

        var uri = request.getRequestURI();
        if (uri.startsWith(OpenFeignConstant.URI_PREFIX)) {
            // rpc调用，忽略数据权限
            return false;
        }

        return true;
    }

    private <T extends BaseDataSecurityRuleDTO> List<T> filterRule(HttpServletRequest request, String businessObjectCode,
                                                                   String businessOperationCode, List<T> ruleList) {
        log.info("businessObject:{}, businessObjectOperation:{}", businessObjectCode, businessOperationCode);

        // 细粒度：菜单与业务操作
        var filteredRuleList = this.filterByMenuOperation(ruleList, request, businessObjectCode, businessOperationCode);
        if (!filteredRuleList.isEmpty()) {
            log.info("matched by MenuOperation");
            return filteredRuleList;
        }

        if (properties.getCompatible() == CompatibleModeEnum.OLDEST) {
            // 老的不支持业务对象
            return Collections.emptyList();
        }
        // 基于业务对象的数据权限
        if (CharSequenceUtil.isBlank(businessObjectCode)) {
            return Collections.emptyList();
        }

        // 中粒度：业务对象操作
        filteredRuleList = this.filterByBusinessOperation(ruleList, request, businessObjectCode, businessOperationCode);
        if (!filteredRuleList.isEmpty()) {
            log.info("matched by BusinessOperation");
            return filteredRuleList;
        }

        // 大粒度：业务对象
        filteredRuleList = this.filterByBusinessObject(ruleList, businessObjectCode);
        if (!filteredRuleList.isEmpty()) {
            log.info("matched by BusinessObject");
            return filteredRuleList;
        }

        log.info("No permission rules were matched：{}", request.getRequestURI());
        return filteredRuleList;
    }

    private <T extends BaseDataSecurityRuleDTO> List<T> filterByMenuOperation(List<T> ruleList, HttpServletRequest request,
                                                                              String businessObjectCode, String businessOperationCode) {
        var menuCode = request.getHeader(WebConstant.HEADER_MENU_CODE);
        log.info("menuCode：{}", menuCode);
        if (CharSequenceUtil.isBlank(menuCode)) {
            return Collections.emptyList();
        }

        var compatible = properties.getCompatible();
        return ruleList.stream()
                .filter(t -> {
                    if (compatible == CompatibleModeEnum.OLDEST) {
                        // 老的模式，则排除掉新的权限配置
                        if (StringUtils.hasText(t.getPermissionType())) {
                            return false;
                        }
                    } else if (compatible == CompatibleModeEnum.LATEST) {
                        // 新的模式，则排除掉老的配置
                        if (!StringUtils.hasText(t.getPermissionType())) {
                            return false;
                        }
                    }

                    // 菜单
                    if (!menuCode.equals(t.getMenusCode())) {
                        return false;
                    }

                    // 新的权限规则
                    if (StringUtils.hasText(t.getPermissionType())) {
                        if (!DataPermissionType.MENU_OPERATION_RULE.name().equals(t.getPermissionType())) {
                            return false;
                        }
                        if (!t.getBusinessObjectCode().equals(businessObjectCode)) {
                            return false;
                        }
                        if (CharSequenceUtil.isBlank(businessOperationCode)) {
                            return false;
                        }
                    }

                    return this.isMatchRequestForOperation(t, request, businessOperationCode);
                }).collect(Collectors.toList());
    }

    private <T extends BaseDataSecurityRuleDTO> List<T> filterByBusinessOperation(List<T> ruleList, HttpServletRequest request,
                                                                                  String businessObjectCode, String businessOperationCode) {
        if (!StringUtils.hasText(businessObjectCode)) {
            return Collections.emptyList();
        }

        return ruleList.stream()
                .filter(t -> {
                    // 权限类型
                    if (!DataPermissionType.BUSINESS_OPERATION_RULE.name().equals(t.getPermissionType())) {
                        return false;
                    }

                    // 业务对象
                    if (!businessObjectCode.equals(t.getBusinessObjectCode())) {
                        return false;
                    }

                    return this.isMatchRequestForOperation(t, request, businessOperationCode);
                }).collect(Collectors.toList());
    }

    private <T extends BaseDataSecurityRuleDTO> List<T> filterByBusinessObject(List<T> ruleList,
                                                                               String businessObjectCode) {
        // 业务对象
        if (!StringUtils.hasText(businessObjectCode)) {
            return Collections.emptyList();
        }

        return ruleList.stream()
                .filter(t -> {
                    // 权限类型
                    if (!DataPermissionType.BUSINESS_OBJECT_RULE.name().equals(t.getPermissionType())) {
                        return false;
                    }

                    // 业务对象
                    if (!businessObjectCode.equals(t.getBusinessObjectCode())) {
                        return false;
                    }

                    return true;
                }).collect(Collectors.toList());
    }

    private String queryBusinessObjectCodeByBusinessOperationCode(String businessOperationCode) {
        var businessOperation = roleDataPermissionRuleCacheInterface.getBusinessOperationByBusinessOperationCode(businessOperationCode);
        Assert.notNull(businessOperation, "业务操作[" + businessOperationCode + "]不存在");
        return businessOperation.getBusinessObjectCode();
    }

    private String obtainBusinessObjectCode(HttpServletRequest request) {
        // 优先从前端获取
        var businessObjectCode = request.getHeader(WebConstant.HEADER_BUSINESS_OBJECT);
        if (StringUtils.hasText(businessObjectCode)) {
            return businessObjectCode;
        }

        var businessObject = (BusinessObject) request.getAttribute(WebConstant.ATTRIBUTE_BUSINESS_OBJECT);
        var businessOperation = (BusinessObjectOperation) request.getAttribute(WebConstant.ATTRIBUTE_BUSINESS_OPERATION);
        if (businessOperation == null && businessObject == null) {
            return null;
        }

        // 从业务操作获取
        if (businessOperation != null) {
            if (BusinessObjectOperation.BUSINESS_OBJECT_NAME_DEFAULT.equals(businessOperation.businessObjectType())) {
                // 取业务对象的
                return businessObject == null ? null : businessObject.businessType().split(":")[0];
            }
            return businessOperation.businessObjectType().split(":")[0];
        }

        // 最后取业务对象的
        return businessObject.businessType().split(":")[0];
    }

    private String obtainBusinessOperationCode(HttpServletRequest request) {
        // 优先从前端获取
        var businessOperationCode = request.getHeader(WebConstant.HEADER_BUSINESS_OPERATION);
        if (StringUtils.hasText(businessOperationCode)) {
            return businessOperationCode;
        }

        // 从方法上的注解获取
        var businessOperation = (BusinessObjectOperation) request.getAttribute(WebConstant.ATTRIBUTE_BUSINESS_OPERATION);
        if (businessOperation == null) {
            return null;
        }

        return StringUtils.hasText(businessOperation.operationCode()) ? businessOperation.operationCode() : null;
    }

    private <T extends BaseDataSecurityRuleDTO> boolean isMatchRequestForOperation(@NotNull T rule,
                                                                                   @NotNull HttpServletRequest request,
                                                                                   String operationCode) {
        // 兼容老的：前端传递api编码
        var apiCode = request.getHeader(WebConstant.HEADER_API_CODE);
        if (StringUtils.hasText(apiCode)) {
            // 兼容老的
            return apiCode.equals(rule.getApiPermissionCode());
        }

        // 根据业务操作匹配
        if (rule.getApiPermissionCode().equals(operationCode)) {
            return true;
        }

        // 请求方式
        if (!request.getMethod().equalsIgnoreCase(rule.getApiPermissionRequestType())) {
            return false;
        }

        var uri = request.getRequestURI();
        // 路径参数
        var pathVariables = HttpServletUtil.getPathVariables(request);
        if (pathVariables.isEmpty()) {
            // 没有路径参数，则直接匹配
            return uri.equals(rule.getApiPermissionPath());
        }
        // 路径参数转换
        var path = StringUtils.hasText(rule.getPermissionType()) ? StrUtil.format(rule.getApiPermissionPath(), pathVariables)
                : MessageFormat.format(rule.getApiPermissionPath(), pathVariables.values());
        return uri.equals(path);
    }
}
