package com.elitescloud.boot.datasecurity.jpa.strategy;

import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.jpa.strategy.intgerStrategy.IntegerEqualStrategy;
import com.elitescloud.boot.datasecurity.jpa.strategy.intgerStrategy.IntegerInStrategy;
import com.elitescloud.boot.datasecurity.jpa.strategy.longStrategy.LongEqualStrategy;
import com.elitescloud.boot.datasecurity.jpa.strategy.longStrategy.LongInStrategy;
import com.elitescloud.boot.datasecurity.jpa.strategy.stringStrategy.StringContainStrategy;
import com.elitescloud.boot.datasecurity.jpa.strategy.stringStrategy.StringEqualStrategy;
import com.elitescloud.boot.datasecurity.jpa.strategy.stringStrategy.StringInStrategy;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;

import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.PathBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

/**
 * @author : chen.niu
 * @description : 规则策略管理
 * @date : 2023/10/14 17:22
 */

@Slf4j
public  class RuleStrategyManager {
    private final static  Map<String, RuleStrategy> strategies = new HashMap<>();


    static {
        // 在这里注册所有的策略实现
        strategies.put(TypeNames.LONG+ DprRuleConditionEnum.Equal.name(), new LongEqualStrategy());
        strategies.put(TypeNames.LONG+ DprRuleConditionEnum.InList.name(), new LongInStrategy());

        strategies.put(TypeNames.STRING+ DprRuleConditionEnum.Equal.name(), new StringEqualStrategy());
        strategies.put(TypeNames.STRING+ DprRuleConditionEnum.InList.name(), new StringInStrategy());
        strategies.put(TypeNames.STRING+ DprRuleConditionEnum.Contain.name(), new StringContainStrategy());


        strategies.put(TypeNames.INTEGER+ DprRuleConditionEnum.Equal.name(), new IntegerEqualStrategy());
        strategies.put(TypeNames.INTEGER+ DprRuleConditionEnum.InList.name(), new IntegerInStrategy());
//        strategies.put(TypeNames.INTEGER+ DprRuleConditionEnum.NotEqual.name(), new IntegerEqualStrategy());
//        strategies.put(TypeNames.INTEGER+ DprRuleConditionEnum.InList.name(), new IntegerInStrategy());

    }

    private RuleStrategyManager() {
        // 私有构造方法，防止外部实例化
    }
    public static Predicate getPathAutomatically(PathBuilder<?> entityPath,
                                                   SysDprRoleApiDataRuleListQueryDTO sysRule) {
        try {
            //规则字段
            String fieldName = sysRule.getDprRuleField();
            //规则条件
            String ruleCondition = sysRule.getDprRuleCondition();
            String ruleValue=sysRule.getDprRuleValue();
            return RuleStrategyManager.getPredicate(entityPath, fieldName, ruleCondition, ruleValue);
        } catch (Exception e) {
            throw new IllegalArgumentException("数据权限异常: " , e);
        }

    }
    private static Predicate getPredicate(PathBuilder<?> entityPath, String fieldName, String ruleCondition, String ruleValue) {
        Assert.hasText(ruleCondition, "条件为空");
        String fieldType = getFieldType(entityPath.getType(), fieldName);
        String key = fieldType + ruleCondition;
        RuleStrategy strategy = strategies.get(key);
        if (strategy == null) {
            throw new IllegalStateException("没有对应规则的key: " + key);
        }
        return strategy.applyRule(entityPath, fieldName, ruleValue);
    }

    private static String getFieldType(Class<?> entityClass, String fieldName) {
        try {
            Field field = getFieldIncludingParentClasses(entityClass, fieldName);
            return field.getType().getSimpleName();
        } catch (NoSuchFieldException e) {
            throw new IllegalStateException("字段没有找到: " + fieldName, e);
        }
    }

    private static Field getFieldIncludingParentClasses(Class<?> clazz, String fieldName) throws NoSuchFieldException {
        try {
            return clazz.getDeclaredField(fieldName);
        } catch (NoSuchFieldException e) {
            Class<?> superClass = clazz.getSuperclass();
            if (superClass == null) {
                throw e;
            } else {
                return getFieldIncludingParentClasses(superClass, fieldName);
            }
        }
    }
}
