package com.elitescloud.boot.datasecurity.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.constant.WebConstant;
import com.elitescloud.boot.datasecurity.common.DataSecurityUtil;
import com.elitescloud.boot.datasecurity.config.DataSecurityProperties;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.filter.UdcFilter;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.system.constant.DprRoleConstant;
import com.elitescloud.cloudt.system.dto.SysUdcDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据权限过滤UDC值.
 *
 * @author Kaiser（wang shao）
 * @date 2023/11/27
 */
@Slf4j
public class DataSecurityUdcFilter implements UdcFilter {
    private final DataSecurityProperties properties;

    public DataSecurityUdcFilter(DataSecurityProperties properties) {
        this.properties = properties;
    }

    @Override
    public void doFilter(List<SysUdcDTO> udcList) {
        if (!this.support()) {
            return;
        }

        // 筛选需要过滤的UDC
        var udcListMap = this.filterUdc(udcList);
        if (udcListMap.isEmpty()) {
            return;
        }

        // 获取授权的udc值
        var authUdcValueList = this.obtainAuthedUdcValue();
        if (authUdcValueList.isEmpty()) {
            return;
        }

        var udcValuesMap = authUdcValueList.stream().collect(Collectors.groupingBy(t -> t.getAppCode() + ":" + t.getUdcCode()));
        if (log.isDebugEnabled()) {
            log.debug("可过滤的UDC：{}", String.join(", ", udcValuesMap.keySet()));
        }
        List<FilteredUdc> filteredValueList = null;
        List<SysUdcDTO.UdcValue> resultValueList = null;
        for (SysUdcDTO udcDTO : udcList) {
            // 获取需要过滤的udc
            filteredValueList = udcValuesMap.get(udcDTO.getAppCode() + ":" + udcDTO.getUdcCode());
            if (filteredValueList == null) {
                continue;
            }
            // 需要过滤的udc值
            Set<String> includeValues = new HashSet<>();
            Set<String> excludeValues = new HashSet<>();
            for (FilteredUdc filteredUdc : filteredValueList) {
                if (CollUtil.isNotEmpty(filteredUdc.getUdcValueInclude())) {
                    includeValues.addAll(filteredUdc.getUdcValueInclude());
                }
                if (CollUtil.isNotEmpty(filteredUdc.getUdcValueExclude())) {
                    excludeValues.addAll(filteredUdc.getUdcValueExclude());
                }
            }

            resultValueList = udcDTO.getValueList().stream()
                    .filter(t -> {
                        if (!excludeValues.isEmpty()) {
                            return includeValues.isEmpty() ? !excludeValues.contains(t.getUdcValueCode()) :
                                    !excludeValues.contains(t.getUdcValueCode()) && includeValues.contains(t.getUdcValueCode());
                        }
                        if (!includeValues.isEmpty()) {
                            return includeValues.contains(t.getUdcValueCode());
                        }
                        return true;
                    }).collect(Collectors.toList());
            udcDTO.setValueList(resultValueList);
        }
    }

    private boolean support() {
        if (Boolean.FALSE.equals(properties.getEnabled()) || Boolean.FALSE.equals(properties.getUdc().getEnabled())) {
            log.debug("数据权限已禁用");
            return false;
        }

        // 是否来自前端的查询
        var request = HttpServletUtil.currentRequest();
        if (request == null) {
            return false;
        }
        var support = (Boolean) request.getAttribute(WebConstant.ATTRIBUTE_UDC_FILTER_SUPPORT);
        if (!Boolean.TRUE.equals(support)) {
            return false;
        }
        var menuCode = request.getHeader(WebConstant.HEADER_MENU_CODE);
        if (!StringUtils.hasText(menuCode)) {
            return false;
        }

        var user = SecurityContextUtil.currentUser();
        if (user == null || CollUtil.isEmpty(user.getUser().getRoles())) {
            // 无角色，不需要权限过滤
            return false;
        }

        return true;
    }

    private Map<String, List<SysUdcDTO>> filterUdc(List<SysUdcDTO> udcList) {
        if (CollUtil.isEmpty(udcList)) {
            return Collections.emptyMap();
        }

        // 筛选需要过滤的UDC
        Map<String, List<SysUdcDTO>> udcListMap = new HashMap<>(udcList.size());
        for (SysUdcDTO udcDTO : udcList) {
            if (udcDTO == null || CollUtil.isEmpty(udcDTO.getValueList())) {
                continue;
            }
            udcListMap.computeIfAbsent(udcDTO.getUdcCode(), k -> new ArrayList<>(4))
                    .add(udcDTO);
        }
        return udcListMap;
    }

    private List<FilteredUdc> obtainAuthedUdcValue() {
        // 获取用户的权限规则
        var userRoleDpr = DataSecurityUtil.getAllDataPermission();
        if (userRoleDpr == null || CollUtil.isEmpty(userRoleDpr.getSysDprRoleApiDataRuleListQueryDTO())) {
            return Collections.emptyList();
        }

        // 获取过滤的菜单编码
        var request = HttpServletUtil.currentRequest();
        Assert.notNull(request, "request为空");
        var menuCode = (String) request.getHeader(WebConstant.HEADER_MENU_CODE);
        return userRoleDpr.getSysDprRoleApiDataRuleListQueryDTO().stream()
                .filter(t ->
                        menuCode.equals(t.getMenusCode())
                                && DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name().equals(t.getDprRuleValueType())
                                && DprRoleConstant.DATA_SET_TYPE_UDC.equals(t.getDataSet())
                                && CharSequenceUtil.isAllNotBlank(t.getDprRuleValue(), t.getBs1(), t.getBs2()))
                .map(t -> {
                    FilteredUdc filteredUdc = new FilteredUdc();
                    filteredUdc.setAppCode(t.getBs1());
                    filteredUdc.setUdcCode(t.getBs2());

                    var values = Arrays.asList(t.getDprRuleValue().split(","));
                    if (DprRuleConditionEnum.Equal.name().equals(t.getDprRuleCondition()) ||
                            DprRuleConditionEnum.InList.name().equals(t.getDprRuleCondition())) {
                        filteredUdc.setUdcValueInclude(values);
                        filteredUdc.setUdcValueExclude(Collections.emptyList());
                    } else if (DprRuleConditionEnum.NotEqual.name().equals(t.getDprRuleCondition()) ||
                            DprRuleConditionEnum.NotIn.name().equals(t.getDprRuleCondition())) {
                        filteredUdc.setUdcValueInclude(Collections.emptyList());
                        filteredUdc.setUdcValueExclude(values);
                    }

                    return filteredUdc;
                }).collect(Collectors.toList());
    }

    static class FilteredUdc {
        private String appCode;

        private String udcCode;

        private List<String> udcValueInclude;

        private List<String> udcValueExclude;

        public String getAppCode() {
            return appCode;
        }

        public void setAppCode(String appCode) {
            this.appCode = appCode;
        }

        public String getUdcCode() {
            return udcCode;
        }

        public void setUdcCode(String udcCode) {
            this.udcCode = udcCode;
        }

        public List<String> getUdcValueInclude() {
            return udcValueInclude;
        }

        public void setUdcValueInclude(List<String> udcValueInclude) {
            this.udcValueInclude = udcValueInclude;
        }

        public List<String> getUdcValueExclude() {
            return udcValueExclude;
        }

        public void setUdcValueExclude(List<String> udcValueExclude) {
            this.udcValueExclude = udcValueExclude;
        }
    }
}
