package com.elitescloud.boot.datasecurity.dpr.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.datasecurity.dpr.service.RoleDataPermissionRuleCacheService;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.model.dto.SysBusinessOperationDTO;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.threadpool.common.ThreadPoolHolder;
import com.elitescloud.cloudt.system.constant.SysCacheConstant;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import com.elitescloud.cloudt.system.provider.extend.SysBusinessObjectRpcService;
import com.elitescloud.cloudt.system.service.RoleAppApiDataPermissionRpcService;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

/**
 * 角色数据权限的规则的缓存处理
 *
 * @author : chen
 * @date 2022-11-25 10:53
 */
@Slf4j
public class RoleDataPermissionRuleCacheServiceImpl implements RoleDataPermissionRuleCacheService {
    private final RedisUtils redisUtils;

    private final RoleAppApiDataPermissionRpcService roleAppApiDataPermissionRpcService;
    private final SysBusinessObjectRpcService businessObjectRpcService;
    private final TaskExecutor executor;
    /**
     * 数据权限缓存token前缀拼接字符串
     */
    public final static String TOKEN_PREFIX = "DRP_TOKEN_";
    /**
     * redis缓存时间
     */
    public final static Integer REDIS_UTILS_TIME = 120;


    /**
     * 本地缓存配置 最大缓存数量
     */
    public final static Integer CACHE_MAXIMUM_SIZE = 3000;
    /**
     * 本地缓存配置 缓存时间
     */
    public final static Integer CACHE_EXPIRE_AFTER_WRITE = 5;
    /**
     * 本地缓存配置对象 专门用于存储token和当前用户角色权限信息
     */
    private static final Cache<String, SysDprRoleApiRowColumnRuleDTO> TOKEN_DPR_LOCAL_CACHE = Caffeine.newBuilder()
            .maximumSize(CACHE_MAXIMUM_SIZE)
            .expireAfterWrite(Duration.ofMinutes(CACHE_EXPIRE_AFTER_WRITE))
            .build();

    public RoleDataPermissionRuleCacheServiceImpl(RedisUtils redisUtils,
                                                  RoleAppApiDataPermissionRpcService roleAppApiDataPermissionRpcService,
                                                  SysBusinessObjectRpcService businessObjectRpcService,
                                                  TaskExecutor taskExecutor) {
        this.redisUtils = redisUtils;
        this.roleAppApiDataPermissionRpcService = roleAppApiDataPermissionRpcService;
        this.businessObjectRpcService = businessObjectRpcService;
        this.executor = taskExecutor;
    }

    @Override
    public SysDprRoleApiRowColumnRuleDTO roleDataPermissionRuleRpc(String token) {

        try {
            var apiResult
                    = roleAppApiDataPermissionRpcService.getRoleAppApiDataRulePermission();

            if (!apiResult.isSuccess()) {
                throw new BusinessException("远程调用获取角色数据权限异常：" + apiResult.getMsg());
            }
            if (apiResult.getData() == null) {
                log.debug("数据权限：当前token账号，没有任何数据权限配置。返回NULL。   token:{},data:{}", token, apiResult.getData());
                return null;
            }
            //本地缓存
            tokenDprLocalCacheSave(token, apiResult.getData());
            if (apiResult.getData() != null) {
                //异步进行redis缓存
                CompletableFuture.runAsync(() -> {
                    //redis存储
                    if (!tokenRedisCacheSave(token, apiResult.getData())) {
                        log.error("数据权限：用户数据权限Redis缓存失败，请尽快排查 token:{},data:{}", token, apiResult.getData());
                    } else {
                        log.debug("数据权限：Token redis缓存成功 token:{},data:{}", token, apiResult.getData());
                    }
                }, executor);
            } else {
                log.debug("数据权限：data数据空，跳过Redis缓存。");
            }
            return apiResult.getData();
        } catch (Exception e) {
            log.error("获取数据权限异常：{}", e.getMessage());
            throw new BusinessException("获取数据权限异常：{}" + e.getMessage());
        }
    }

    @Override
    public void tokenDprLocalCacheSave(String token, SysDprRoleApiRowColumnRuleDTO roleRuleGroupDtoList) {
        //本地缓存
        TOKEN_DPR_LOCAL_CACHE.put(token, roleRuleGroupDtoList);
    }

    @Override
    public Boolean tokenRedisCacheSave(String token, SysDprRoleApiRowColumnRuleDTO roleRuleGroupDtoList) {
        if (!redisUtils.set(TOKEN_PREFIX + token, roleRuleGroupDtoList, REDIS_UTILS_TIME, TimeUnit.MINUTES)) {
            log.error("数据权限：redis缓存权限信息失败 token:{}  -  data:{}", token, roleRuleGroupDtoList);
            return false;
        } else {
            return true;
        }
    }


    @Override
    public Optional<SysDprRoleApiRowColumnRuleDTO> getTokenDprLocalCache(String token) {
        //  从本地缓存查询
        var permissions = TOKEN_DPR_LOCAL_CACHE.getIfPresent(token);
        if (permissions != null) {
            return Optional.of(permissions);
        } else {
            log.debug("数据权限：没有读取到本地缓存 ：" + token);
            return Optional.empty();
        }
    }

    @Override
    public Optional<SysDprRoleApiRowColumnRuleDTO> getTokenDprRedisCache(String token) {
        SysDprRoleApiRowColumnRuleDTO sysDprRoleApiRuleGroupDTOList;
        sysDprRoleApiRuleGroupDTOList = (SysDprRoleApiRowColumnRuleDTO) redisUtils.get(TOKEN_PREFIX + token);
        if (sysDprRoleApiRuleGroupDTOList != null) {

            //本地缓存存储 避免过期  redis缓存时间较长
            tokenDprLocalCacheSave(token, sysDprRoleApiRuleGroupDTOList);

            return Optional.of(sysDprRoleApiRuleGroupDTOList);
        } else {
            log.debug("数据权限：没有读取到redis数据权限" + token);
            return Optional.empty();
        }
    }

    @Override
    public SysBusinessOperationDTO getBusinessOperationByBusinessOperationCode(String businessOperationCode) {
        // 先从缓存获取
        SysBusinessOperationDTO operationDTO = null;
        try {
            operationDTO = CompletableFuture.supplyAsync(() -> (SysBusinessOperationDTO) redisUtils.hget(SysCacheConstant.BUSINESS_OPERATION_ALL_HASH, businessOperationCode))
                    .get();
        } catch (Exception e) {
            throw new BusinessException("根据业务操作编码[" + businessOperationCode + "]获取业务操作信息异常", e);
        }
        if (operationDTO != null) {
            return operationDTO;
        }

        // 调用RPC接口获取
        var queryResult = businessObjectRpcService.allOperation().computeData();
        if (CollUtil.isEmpty(queryResult)) {
            return null;
        }

        for (var sysBusinessOperationDTO : queryResult) {
            if (businessOperationCode.equals(sysBusinessOperationDTO.getOperationCode())) {
                return sysBusinessOperationDTO;
            }
        }
        return null;
    }
}
