package com.elitescloud.boot.datasource.config.support;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.datasource.DataSourceSelector;
import com.elitescloud.boot.datasource.DataSourceSwitchable;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.aspectj.MethodInvocationProceedingJoinPoint;
import org.springframework.util.Assert;

/**
 * 多数据源自动切换拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/1
 */
@Aspect
public class CloudtMultiDataSourceAspect {

    private final DataSourceSwitchable dataSourceSwitchable;

    public CloudtMultiDataSourceAspect(DataSourceSwitchable dataSourceSwitchable) {
        this.dataSourceSwitchable = dataSourceSwitchable;
    }

    @Pointcut("@annotation(com.elitescloud.boot.datasource.DataSourceSelector)")
    private void pointCutMethod() {
    }

    @Pointcut("@within(com.elitescloud.boot.datasource.DataSourceSelector)")
    private void pointCutClass() {
    }

    @Around("pointCutClass() || pointCutMethod()")
    @SuppressWarnings("unchecked")
    public Object cutAround(ProceedingJoinPoint point) throws Throwable {
        // 先获取注解
        DataSourceSelector annotation = obtainAnnotation(point);
        Assert.notNull(annotation, "未获取到Ds注解信息");

        String dsName = CharSequenceUtil.blankToDefault(annotation.value(), annotation.name());
        Assert.hasText(dsName, () -> "未指定数据源名称：" + point.getThis().getClass().getName());
        return dataSourceSwitchable.execute(() -> {
            try {
                return point.proceed();
            } catch (Throwable e) {
                if (e instanceof RuntimeException) {
                    throw (RuntimeException) e;
                }
                throw new RuntimeException(e);
            }
        }, dsName);
    }

    private DataSourceSelector obtainAnnotation(ProceedingJoinPoint point) {
        if (point instanceof MethodInvocationProceedingJoinPoint) {
            // 优先获取方法上的注解
            DataSourceSelector annotation = ((MethodSignature) point.getSignature()).getMethod().getAnnotation(DataSourceSelector.class);

            if (annotation == null) {
                annotation = point.getThis().getClass().getAnnotation(DataSourceSelector.class);
            }

            return annotation;
        }
        return null;
    }
}
