package com.elitescloud.boot.datasource.config.support;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.datasource.DataSourceSwitchable;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.function.Supplier;

/**
 * 多数据源切换器.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/1
 */
@Slf4j
public class CloudtMultiDataSourceSwitcher implements DataSourceSwitchable {

    private final PlatformTransactionManager transactionManager;

    public CloudtMultiDataSourceSwitcher(PlatformTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }

    @Override
    public <T> T execute(Supplier<T> supplier, String dsName) {
        Assert.notNull(supplier, "待执行操作为空");

        // 切换前的数据源名称
        var beforeName = DataSourceContextHolder.getDatasource();
        log.info("switch dataSource：{} -> {}", beforeName, dsName);
        if (CharSequenceUtil.equals(beforeName, dsName)) {
            // 数据源没变
            try {
                return supplier.get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        // 开始切换
        DataSourceContextHolder.setDatasource(dsName);
        String newTransactionName = "cloudt-dynamicDs[" + dsName + "]";
        var transaction = buildTenantTransactionDefinition(newTransactionName);
        try {
            T result = supplier.get();
            // 提交事务
            afterCompletion(transaction, null);
            return result;
        } catch (Exception e) {
            // 回滚事务
            afterCompletion(transaction, e);
            throw new RuntimeException(e);
        } finally {
            // 还原原数据源
            if (StringUtils.hasText(beforeName)) {
                DataSourceContextHolder.setDatasource(beforeName);
            } else {
                DataSourceContextHolder.clearDatasource();
            }
        }
    }

    private void afterCompletion(TransactionStatus transactionStatus, Exception exp) {
        if (exp == null) {
            // 没有异常，则正常提交
            transactionManager.commit(transactionStatus);
            return;
        }

        //出现异常则回滚
        transactionManager.rollback(transactionStatus);
        log.error("执行业务异常：", exp);
    }

    private TransactionStatus buildTenantTransactionDefinition(String newTransactionName) {
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        definition.setName(newTransactionName);
        return transactionManager.getTransaction(definition);
    }
}
