package com.elitescloud.boot.datasource.config.support;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.datasource.config.CloudtMultiDataSourceProperties;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.core.env.Environment;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

/**
 * 多数据源.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/1
 */
@Slf4j
public class MultiDataSource implements DataSource, EnvironmentAware, InitializingBean {
    private Environment environment;
    private final CloudtMultiDataSourceProperties properties;

    private final Map<String, DataSource> dataSourceMap = new HashMap<>(8);
    ;
    private DataSource defaultDatasource = null;

    public MultiDataSource(CloudtMultiDataSourceProperties properties) {
        this.properties = properties;
    }

    @Override
    public Connection getConnection() throws SQLException {
        return getDataSource().getConnection();
    }

    @Override
    public Connection getConnection(String username, String password) throws SQLException {
        return getDataSource().getConnection(username, password);
    }

    @Override
    public PrintWriter getLogWriter() throws SQLException {
        return getDataSource().getLogWriter();
    }

    @Override
    public void setLogWriter(PrintWriter out) throws SQLException {
        getDataSource().setLogWriter(out);
    }

    @Override
    public void setLoginTimeout(int seconds) throws SQLException {
        getDataSource().setLoginTimeout(seconds);
    }

    @Override
    public int getLoginTimeout() throws SQLException {
        return getDataSource().getLoginTimeout();
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return getDataSource().unwrap(iface);
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return getDataSource().isWrapperFor(iface);
    }

    @Override
    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        return getDataSource().getParentLogger();
    }


    @Override
    public void setEnvironment(@NonNull Environment environment) {
        this.environment = environment;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.init();
    }

    private void init() {
        Assert.isTrue(!properties.getMulti().isEmpty(), "未发现多数据的配置");
        var def = properties.getDefaultDatasource();
        Assert.hasText(def, "请配置默认数据源的名称");

        var binder = Binder.get(environment);
        // 加载默认数据源配置
        var defaultProperties = this.obtainDefaultProperties(binder);
        // 加载多数据源配置
        loadMultiDataSourceProperties(binder, defaultProperties);
        defaultDatasource = dataSourceMap.get(def);

        Assert.notNull(defaultDatasource, "默认数据源" + def + "不存在");
        log.info("default datasource is: {}", def);
    }

    private void loadMultiDataSourceProperties(Binder binder, Properties defaultProperties) {
        var dataSourceNames = properties.getMulti().keySet();

        for (String datasourceName : dataSourceNames) {
            log.info("start loading datasource: {}", datasourceName);
            // 绑定数据源
            var prefix = CloudtMultiDataSourceProperties.CONFIG_PREFIX + ".multi." + datasourceName;
            var bindResult = binder.bind(prefix, DataSourceProperties.class);
            if (!bindResult.isBound()) {
                throw new IllegalStateException("数据源" + datasourceName + "加载失败，请确认已配置");
            }
            var properties = bindResult.get();
            var datasource = properties.initializeDataSourceBuilder().type(HikariDataSource.class).build();
            // hikari配置
            var hikariProperties = hikariProperties(binder, prefix, defaultProperties);
            if (hikariProperties != null) {
                datasource.setDataSourceProperties(hikariProperties);
            }
            datasource.setPoolName(CharSequenceUtil.blankToDefault(datasource.getPoolName(), "cloudt-ds-" + datasourceName));

            dataSourceMap.put(datasourceName, datasource);
        }
    }

    private Properties obtainDefaultProperties(Binder binder) {
        Properties defaultProperties = hikariProperties(binder, CloudtMultiDataSourceProperties.CONFIG_PREFIX, null);
        if (defaultProperties != null) {
            return defaultProperties;
        }
        return hikariProperties(binder, "spring.datasource", null);
    }

    private Properties hikariProperties(Binder binder, String prefix, Properties defaultProperties) {
        String name = prefix + ".hikari";
        var bindResult = binder.bind(name, Properties.class);
        if (bindResult.isBound()) {
            Properties properties = bindResult.get();
            if (defaultProperties != null) {
                defaultProperties.forEach(properties::putIfAbsent);
            }
            return properties;
        }
        return defaultProperties;
    }

    private DataSource getDataSource() {
        var ds = DataSourceContextHolder.getDatasource();
        if (StringUtils.hasText(ds)) {
            var datasource = dataSourceMap.get(ds);
            Assert.notNull(datasource, ds + "数据源未配置");
            return datasource;
        }
        return defaultDatasource;
    }
}
