package com.elitescloud.boot.dubbo.filter;

import cn.hutool.core.util.BooleanUtil;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.rpc.*;

/**
 * 租户上下文信息的传递.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/6
 */
@Log4j2
@Activate(group = {"provider", "consumer"})
public class DubboTenantContextFilter implements Filter {

    private static final String ATTACHMENT_KEY_TENANT_ID = "cloudt_tenant_id";
    private static final ThreadLocal<Boolean> NOT_NEED_TENANT = new ThreadLocal<>();

    private TenantClientProvider tenantClientProvider;

    /**
     * 设置不需要租户获取
     */
    public static void setNotNeedTenant() {
        NOT_NEED_TENANT.set(true);
    }

    public static void removeNotNeedTenant() {
        NOT_NEED_TENANT.remove();
    }

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        RpcContext rpcContext = RpcContext.getServiceContext();
        if (rpcContext.getUrl() == null || !tenantClientProvider.enabledTenant()) {
            return invoker.invoke(invocation);
        }

        if (rpcContext.isProviderSide()) {
            // 服务端负责放置租户信息
            SysTenantDTO tenantDTO = null;
            String tenantId = rpcContext.getAttachment(ATTACHMENT_KEY_TENANT_ID);
            if (tenantId != null && tenantId.length() > 0) {
                tenantDTO = tenantClientProvider.getTenant(Long.parseLong(tenantId));
                if (tenantDTO != null) {
                    setTenant(tenantDTO);
                    log.info("当前租户：{}", tenantDTO.getTenantCode());
                }
            }

            try {
                return invoker.invoke(invocation);
            } finally {
                if (tenantDTO != null) {
                    clearTenant();
                }
            }
        } else {
            if (!BooleanUtil.orOfWrap(NOT_NEED_TENANT.get(), TenantSession.getNoTenant())) {
                // 需要租户传递
                SysTenantDTO tenantDTO = tenantClientProvider.getSessionTenant();
                if (tenantDTO != null) {
                    rpcContext.setAttachment(ATTACHMENT_KEY_TENANT_ID, tenantDTO.getId().toString());
                }
            }

            return invoker.invoke(invocation);
        }
    }

    private void setTenant(SysTenantDTO tenant) {
        TenantContextHolder.setCurrentTenant(tenant);
    }

    private void clearTenant() {
        TenantContextHolder.clearCurrentTenant();
    }

    public void setTenantClientProvider(TenantClientProvider tenantClientProvider) {
        this.tenantClientProvider = tenantClientProvider;
    }
}
