package com.elitescloud.boot.excel.config.tmpl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.resp.ExportResultRespVO;
import com.elitescloud.cloudt.system.dto.resp.ImportRateRespVO;
import com.elitescloud.cloudt.system.dto.resp.ImportResultRespVO;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.HttpEntity;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.constraints.NotNull;
import java.util.Map;

/**
 * 数据模板导入导出服务.
 *
 * @author Kaiser（wang shao）
 * @date 2/17/2023
 */
@Api(tags = "数据模板导入导出")
@ResponseBody
@RequestMapping(value = "/cloudt/core/import", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class TmplDataController {

    private final TmplDataService service;

    public TmplDataController(TmplDataService service) {
        this.service = service;
    }

    /**
     * 下载模板文件
     *
     * @param tmplCode 模板编号
     * @return 文件流
     */
    @GetMapping(value = "/{tmplCode}/tmpl", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperation(value = "下载模板文件")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板编号", required = true, paramType = "path"),
    })
    public HttpEntity<StreamingResponseBody> downloadTmpl(@PathVariable("tmplCode") String tmplCode) {
        return service.downloadByCode(tmplCode);
    }

    /**
     * 根据模板导入数据
     *
     * @param tmplCode 模板编号
     * @param file     数据文件
     * @return 导入结果
     */
    @PostMapping(value = "/{tmplCode}/data")
    @ApiOperation(value = "根据模板导入数据")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板编号", required = true, paramType = "path"),
            @ApiImplicitParam(name = "file", value = "数据文件", required = true, paramType = "form", dataTypeClass = MultipartFile.class),
    })
    public ApiResult<ImportResultRespVO> importData(@PathVariable("tmplCode") String tmplCode, @NotNull(message = "导入文件为空") MultipartFile file) {
        return service.importData(tmplCode, file);
    }

    /**
     * 根据模板导出数据
     *
     * @param tmplCode   模板编号
     * @param queryParam 查询参数
     * @return 导出结果
     */
    @PostMapping(value = "/{tmplCode}/export", consumes = MediaType.APPLICATION_JSON_VALUE)
    @ApiOperation(value = "根据模板导出数据")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板编号", required = true, paramType = "path"),
            @ApiImplicitParam(name = "queryParam", value = "查询参数", paramType = "body", dataType = "map"),
    })
    public ApiResult<ExportResultRespVO> exportData(@PathVariable("tmplCode") String tmplCode, @RequestBody(required = false) Map<String, Object> queryParam) {
        return service.exportData(tmplCode, queryParam);
    }

    /**
     * 下载导出的数据文件
     *
     * @param recordId 导出记录ID
     * @return 数据文件流
     */
    @GetMapping(value = "/export/{recordId}/file", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperation(value = "下载导出的数据文件")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "recordId", value = "导出记录ID", required = true),
    })
    public HttpEntity<StreamingResponseBody> downloadExportFile(@PathVariable("recordId") Long recordId) {
        return service.downloadExportFile(recordId);
    }

    /**
     * 查询导入（或导出）进度
     *
     * @param recordId 导入(导出)记录ID
     * @return 进度
     */
    @GetMapping(value = "/{recordId}/rate")
    @ApiOperation(value = "查询导入（或导出）进度")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "recordId", value = "记录标识", required = true),
    })
    public ApiResult<ImportRateRespVO> getRate(@PathVariable("recordId") Long recordId) {
        return service.getRate(recordId);
    }
}
