package com.elitescloud.boot.excel.config.tmpl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.excel.common.DataExport;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.excel.config.tmpl.export.ExportStrategyParam;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.boot.excel.config.tmpl.export.strategy.TmplExportStrategyDelegate;
import com.elitescloud.boot.excel.util.ExcelImportUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.AbstractOrderQueryParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import com.elitescloud.cloudt.system.dto.resp.ExportResultRespVO;
import com.elitescloud.cloudt.system.dto.resp.ImportRateRespVO;
import com.elitescloud.cloudt.system.dto.resp.ImportResultRespVO;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.task.TaskExecutor;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.File;
import java.io.Serializable;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * 文件导入导出服务.
 *
 * @author Kaiser（wang shao）
 * @date 2/17/2023
 */
@Log4j2
class TmplDataService implements ApplicationRunner {

    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private ObjectMapper objectMapper;
    @Autowired
    private DataImportServiceFactory dataImportServiceFactory;
    @Autowired
    private DataExportServiceFactory dataExportServiceFactory;
    @Autowired
    private TmplExportStrategyDelegate exportStrategyDelegate;
    private final FileService<?> fileService;

    private final SystemTmplDataSupport fsmTmplSupport;

    // 临时文件夹
    private File tempDir = null;

    public TmplDataService(FileService<?> fileService, SystemTmplDataSupport fsmTmplSupport) {
        this.fileService = fileService;
        this.fsmTmplSupport = fsmTmplSupport;
        // 初始化临时文件夹
        initTempDir();
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        // 处理系统异常停止的记录状态
        CompletableFuture.runAsync(() -> {
            List<Long> importIds = fsmTmplSupport.getUnFinished();
            if (CollUtil.isNotEmpty(importIds)) {
                RecordResultSaveDTO resultSaveDTO = null;
                SysImportRateDTO rateDTO = null;
                for (Long id : importIds) {
                    rateDTO = fsmTmplSupport.getImportRateFromCache(id);
                    resultSaveDTO = RecordResultSaveDTO.builder()
                            .recordId(id)
                            .success(false)
                            .failMsg("系统异常停止")
                            .build();
                    if (rateDTO != null) {
                        resultSaveDTO.setNumSuc(rateDTO.getCount());
                    }
                    fsmTmplSupport.updateImportResult(resultSaveDTO);
                }
            }
        }, taskExecutor).exceptionally(throwable -> {
            log.warn("处理未导入结束的失败", throwable);
            return null;
        });
    }

    private void initTempDir() {
        tempDir = new File(System.getProperty("java.io.tmpdir"));
        if (tempDir.exists()) {
            return;
        }
        if (!tempDir.mkdirs()) {
            throw new IllegalArgumentException("创建临时文件夹失败：" + tempDir.getAbsolutePath());
        }
    }

    /**
     * 根据模板编号下载模板文件
     *
     * @param code 模板编号
     * @return 模板文件
     */
    public HttpEntity<StreamingResponseBody> downloadByCode(@NotBlank String code) {
        return fsmTmplSupport.downloadByCode(code);
    }

    /**
     * 导入数据
     *
     * @param code     模板编号
     * @param dataFile 数据文件
     * @return 导入结果
     */
    public ApiResult<ImportResultRespVO> importData(@NotBlank String code, @NotNull MultipartFile dataFile) {
        // 导入前校验
        var tmplDTO = fsmTmplSupport.getTmplByCode(code);
        String msg = validateBeforeDeal(tmplDTO);
        if (msg != null) {
            return ApiResult.fail(msg);
        }

        // 保存导入记录
        Long recordId = null;
        try {
            recordId = saveRecord(tmplDTO, dataFile, null);
        } catch (Exception exception) {
            afterImport(tmplDTO, null, null, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导入失败，" + exception.getMessage());
        }

        // 获取数据导入服务
        var dataImportService = dataImportServiceFactory.getDataImportService(code);

        // 开始导入
        ImportResultRespVO respVO = null;
        try {
            respVO = startImport(recordId, tmplDTO, dataFile, dataImportService);
        } catch (Exception exception) {
            log.error("导入失败", exception);
            afterImport(tmplDTO, recordId, 0L, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导入失败" + (exception instanceof BusinessException ? "，" + exception.getMessage() : ""));
        }
        if (Boolean.TRUE.equals(respVO.getSync())) {
            afterImport(tmplDTO, recordId, (long) ObjectUtil.defaultIfNull(respVO.getSyncResult().getNumSuccess(), 0), CharSequenceUtil.join("；", respVO.getSyncResult().getFailRecords()));
        }
        return ApiResult.ok(respVO);
    }

    public ApiResult<ExportResultRespVO> exportData(@NotBlank String code, Map<String, Object> queryParam) {
        // 导入前校验
        var tmplDTO = fsmTmplSupport.getTmplByCode(code);
        String msg = validateBeforeDeal(tmplDTO);
        if (msg != null) {
            return ApiResult.fail(msg);
        }

        // 保存导出记录
        Long recordId = null;
        try {
            recordId = saveRecord(tmplDTO, null, queryParam);
        } catch (Exception exception) {
            afterImport(tmplDTO, null, null, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导出失败，文件服务器异常");
        }

        // 获取数据导出服务
        var dataExportService = dataExportServiceFactory.getDataExportService(code);

        // 开始导出
        AnalyseResult result;
        try {
            result = startExport(recordId, tmplDTO, queryParam, dataExportService);
        } catch (Exception exception) {
            log.error("导出失败", exception);
            afterImport(tmplDTO, recordId, 0L, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导出失败" + (exception instanceof BusinessException ? "，" + exception.getMessage() : ""));
        }

        return ApiResult.ok(new ExportResultRespVO(result.getSync(), recordId, result.getTotal()));
    }

    /**
     * 下載导出文件
     *
     * @param id 记录ID
     * @return 文件
     */
    public HttpEntity<StreamingResponseBody> downloadExportFile(Long id) {
        AtomicInteger times = new AtomicInteger(1);
        String code = TmplDataService.retry(() -> fsmTmplSupport.getRecordFileCode(id), c -> {
            if (CharSequenceUtil.isNotBlank(c)) {
                return true;
            }
            if (times.getAndAdd(1) >= 5) {
                return true;
            }
            return false;
        }, Duration.ofMinutes(1));
        if (CharSequenceUtil.isBlank(code)) {
            log.error("下载导出记录{}的文件失败，文件不存在", id);
            return ResponseEntity.badRequest().build();
        }

        return fileService.download(code, null);
    }

    public ApiResult<ImportRateRespVO> getRate(Long id) {
        SysImportRateDTO rateDTO = null;
        try {
            rateDTO = fsmTmplSupport.getImportRateFromCache(id);
            if (rateDTO == null || ObjectUtil.equals(rateDTO.getTotal(), rateDTO.getCount())) {
                rateDTO = fsmTmplSupport.getImportRate(id);
            }
        } catch (Exception exception) {
            log.error("查询导入进度失败", exception);
            return ApiResult.fail("查询导入进度失败");
        }

        if (rateDTO == null) {
            return ApiResult.fail("导入记录不存在");
        }
        ImportRateRespVO respVO = new ImportRateRespVO();
        respVO.setFinish(rateDTO.getFinish());
        respVO.setTotal(rateDTO.getTotal());
        respVO.setCount(rateDTO.getCount());
        if (ObjectUtil.equals(rateDTO.getTotal(), rateDTO.getCount())) {
            respVO.setFinish(true);
            respVO.setRate("100%");
        } else {
            long count = ObjectUtil.defaultIfNull(rateDTO.getCount(), 0).longValue();
            long total = ObjectUtil.defaultIfNull(rateDTO.getTotal(), 1).longValue();
            respVO.setRate(BigDecimal.valueOf(count * 1.0 / total * 100).setScale(0, RoundingMode.DOWN) + "%");
        }

        return ApiResult.ok(respVO);
    }

    private Long saveRecord(SysTmplDTO tmplDTO, MultipartFile dataFile, Map<String, Object> queryParam) {
        return fsmTmplSupport.saveRecord(tmplDTO.getCode(), dataFile, queryParam);
    }

    private String validateBeforeDeal(SysTmplDTO tmplDTO) {
        if (tmplDTO == null || Boolean.FALSE.equals(tmplDTO.getEnabled())) {
            return "模板不存在或已禁用";
        }
        if (CollUtil.isEmpty(tmplDTO.getAttributes())) {
            return "模板无效";
        }

        boolean export = Boolean.TRUE.equals(tmplDTO.getExport());
        if (export) {
            if (!dataExportServiceFactory.isSupport(tmplDTO.getCode())) {
                return "未发现有效的数据导出服务";
            }
        } else {
            if (!dataImportServiceFactory.isSupport(tmplDTO.getCode())) {
                return "未发现有效的数据导入服务";
            }
        }

        // 限流设置
        if (!fsmTmplSupport.updateLimiter(tmplDTO, true)) {
            return "当前访问用户过多，请稍后再试";
        }

        return null;
    }

    private void afterImport(SysTmplDTO tmplDTO, Long recordId, Long numSuc, String msg) {
        afterImport(tmplDTO, recordId, numSuc, msg, null);
    }

    private void afterImport(SysTmplDTO tmplDTO, Long recordId, Long numSuc, String msg, File file) {
        // 更新导入结果
        if (recordId != null) {
            String fileCode = uploadImportFile(file);
            var importResultDTO = RecordResultSaveDTO.builder()
                    .recordId(recordId)
                    .success(CharSequenceUtil.isBlank(msg))
                    .numSuc(numSuc)
                    .failMsg(msg)
                    .fileCode(fileCode)
                    .build();
            fsmTmplSupport.updateImportResult(importResultDTO);
        } else {
            log.error("更新导入导出记录结果失败：{}", msg);
        }

        // 更新限流信息
        fsmTmplSupport.updateLimiter(tmplDTO, false);
    }

    private String uploadImportFile(File file) {
        if (file != null) {
            var uploadResult = fileService.upload(file);

            // 删除临时文件
            file.delete();
            if (uploadResult.isSuccess() && uploadResult.getData() != null) {
                return uploadResult.getData().getFileCode();
            }
            log.error("上传导入导出结果文件失败：{}", uploadResult);
        }
        return null;
    }

    private List<?> analyseData(SysTmplDTO tmplDTO, MultipartFile dataFile, DataImportServiceFactory.ServiceMetaData dataImportService) {
        try {
            return ExcelImportUtil.instance(dataFile.getInputStream())
                    .headRow(tmplDTO.getHeadRow())
                    .dataType(dataImportService.getDataType(), tmplDTO.getAttributes().get(tmplDTO.getFieldTypeRow() - 1))
                    .readAllSync();
        } catch (Exception e) {
            log.error("解析导入数据失败", e);
            throw new BusinessException("解析导入数据失败");
        }
    }

    private AnalyseResult startExport(Long recordId, SysTmplDTO tmplDTO, Map<String, Object> queryParam, DataExportServiceFactory.ServiceMetaData dataExportService) {
        var exportService = dataExportService.getDataExport();
        var param = convertParam(queryParam, dataExportService.getParamType());
        int pageSize = obtainPageSize(dataExportService.getDataExport());
        param.setCurrent(1);
        param.setSize(pageSize);
        var queryResult = exportService.execute(param, 1, pageSize);
        if (queryResult == null || queryResult.getTotal() == 0) {
            throw new BusinessException("没有符合条件的数据");
        }

        int limit = ObjectUtil.defaultIfNull(tmplDTO.getDataLimitPer(), -1);
        if (limit != -1 && queryResult.getTotal() > limit) {
            throw new BusinessException("每次最多允许导出" + limit + "条");
        }
        // 更新导出总数量
        CompletableFuture.runAsync(() -> fsmTmplSupport.updateImportNum(recordId, queryResult.getTotal()), taskExecutor);

        // 判断是否需要异步
        var sync = tmplDTO.getAsyncThreshold() == null || tmplDTO.getAsyncThreshold() == -1 || queryResult.getTotal() <= tmplDTO.getAsyncThreshold();

        if (sync) {
            exportSync(recordId, tmplDTO, queryResult, param, exportService);

            return new AnalyseResult(queryResult.getTotal(), true);
        }
        exportAsync(recordId, tmplDTO, queryResult, param, exportService);

        return new AnalyseResult(queryResult.getTotal(), false);
    }

    private void exportAsync(Long recordId, SysTmplDTO tmplDTO, PagingVO<Serializable> pageData, AbstractOrderQueryParam queryParam,
                             DataExport<Serializable, AbstractOrderQueryParam> dataExport) {
        // 设置当前的用户认证
        var authentication = SecurityContextHolder.getContext().getAuthentication();

        CompletableFuture.supplyAsync(() -> {
                    if (authentication != null) {
                        SecurityContextHolder.getContext().setAuthentication(authentication);
                    }
                    return write2Excel(false, recordId, tmplDTO, pageData, queryParam, dataExport);
                }, taskExecutor)
                .whenComplete((ret, throwable) -> {
                    fsmTmplSupport.removeRate(recordId);
                    if (throwable == null) {
                        afterImport(tmplDTO, recordId, ret, null);
                    } else {
                        afterImport(tmplDTO, recordId, 0L, ExceptionUtil.getRootCause(throwable).getMessage());
                        log.error("导出数据时出现异常：", throwable);
                    }
                });
    }

    private int obtainPageSize(DataExport<Serializable, AbstractOrderQueryParam> dataExportService) {
        Integer pageSize = dataExportService.pageSize();
        if (pageSize == null || pageSize < 1 || pageSize > 1000) {
            // 默认页大小
            return 500;
        }

        return pageSize;
    }

    private int obtainStepSize(DataImport<Serializable> dataImport) {
        Integer stepSize = dataImport.stepSize();
        if (stepSize == null || stepSize < 1 || stepSize > 1000) {
            // 默认页大小
            return 10;
        }

        return stepSize;
    }

    private void exportSync(Long recordId, SysTmplDTO tmplDTO,
                            PagingVO<Serializable> pageData, AbstractOrderQueryParam queryParam, DataExport<Serializable, AbstractOrderQueryParam> dataExport) {
        Long num = write2Excel(true, recordId, tmplDTO, pageData, queryParam, dataExport);
        afterImport(tmplDTO, recordId, num, null);
    }

    private Long write2Excel(boolean sync, Long recordId, SysTmplDTO tmplDTO, PagingVO<Serializable> pageData,
                             AbstractOrderQueryParam queryParam, DataExport<Serializable, AbstractOrderQueryParam> dataExport) {
        var param = new ExportStrategyParam()
                .setSync(sync)
                .setImportId(recordId)
                .setTmplDTO(tmplDTO)
                .setFirstPageData(pageData)
                .setQueryParam(queryParam)
                .setDataExport(dataExport);
        return exportStrategyDelegate.export(param);
    }

    private <E extends Serializable> E convertParam(Map<String, Object> queryParam, Class<E> paramType) {
        try {
            return objectMapper.convertValue(queryParam, paramType);
        } catch (IllegalArgumentException e) {
            throw new BusinessException("转换查询参数失败，请检查参数格式", e);
        }
    }

    private ImportResultRespVO startImport(Long recordId, SysTmplDTO tmplDTO, MultipartFile dataFile,
                                           DataImportServiceFactory.ServiceMetaData dataImportService) {
        // 判断是否需要异步
        List dataList = analyseData(tmplDTO, dataFile, dataImportService);
        var total = dataList.size();

        if (total == 0) {
            throw new BusinessException("导入数据为空");
        }
        if (tmplDTO.getDataLimitPer() != -1 && total > tmplDTO.getDataLimitPer()) {
            throw new BusinessException("每次最多允许导入" + tmplDTO.getDataLimitPer() + "条");
        }

        // 更新导入总数量
        fsmTmplSupport.updateImportNum(recordId, (long) total);

        var sync = tmplDTO.getAsyncThreshold() == null || tmplDTO.getAsyncThreshold() == -1 || total <= tmplDTO.getAsyncThreshold();
        if (sync) {
            // 需要同步处理
            var syncResult = dataImportService.getDataImport().execute(dataList, tmplDTO.getHeadRow() + 1);
            return ImportResultRespVO.builder().sync(true).syncResult(syncResult).build();
        }

        // 需要异步处理
        CompletableFuture.supplyAsync(() -> importAsync(tmplDTO, recordId, dataImportService.getDataImport(), dataList), taskExecutor)
                .whenComplete((ret, throwable) -> {
                    fsmTmplSupport.removeRate(recordId);

                    if (throwable == null) {
                        // 导入成功
                        afterImport(tmplDTO, recordId, (long) ret.getNumSuccess(), CharSequenceUtil.join("；", ret.getFailRecords()));
                    } else {
                        // 导入失败
                        String failMsg = ret != null && ret.getFailRecords() != null ? CharSequenceUtil.join("；", ret.getFailRecords()) : "";

                        afterImport(tmplDTO, recordId, 0L, failMsg + "；" + ExceptionUtil.getRootCause(throwable).getMessage());
                        log.error("导入数据时出现异常：", throwable);
                    }
                });
        return ImportResultRespVO.builder().sync(false).asyncResult(ImportResultRespVO.AsyncResult.builder().importId(recordId).build()).build();
    }

    private ImportResultRespVO.SyncResult importAsync(SysTmplDTO tmplDTO, Long recordId, DataImport<Serializable> dataImportService, List<Serializable> dataList) {
        int total = dataList.size();
        var size = obtainStepSize(dataImportService);
        var start = 0;
        var numSuc = 0;
        ImportResultRespVO.SyncResult tempResult = null;
        List<String> fail = new ArrayList<>(1024);
        for (var i = 0; ; i++) {
            start = i * size;

            try {
                tempResult = dataImportService.execute(dataList.subList(start, Math.min(start + size, total)),
                        tmplDTO.getHeadRow() + 1 + start);
            } catch (Exception exception) {
                log.error("导入失败", exception);
                fail.add(String.format("total：%s，start:%s，exception:%s", total, start, exception.getMessage()));
            }
            if (tempResult != null) {
                numSuc += ObjectUtil.defaultIfNull(tempResult.getNumSuccess(), 0);
                if (CollUtil.isNotEmpty(tempResult.getFailRecords())) {
                    fail.addAll(tempResult.getFailRecords());
                }
            }

            var finish = start + size >= total;
            fsmTmplSupport.storeRate(recordId, SysImportRateDTO.builder()
                    .finish(finish)
                    .total((long) total)
                    .count((long) numSuc)
                    .tmplCode(dataImportService.getTmplCode())
                    .build());
            if (finish) {
                break;
            }
        }

        return ImportResultRespVO.SyncResult.builder().total(total).numSuccess(numSuc).failRecords(fail).build();
    }

    /**
     * 重试
     *
     * @param supplier  功能
     * @param predicate 判断是否满足
     * @param timeout   超时时间
     * @param <T>       结果类型
     * @return 结果
     */
    private static <T> T retry(Supplier<T> supplier, Predicate<T> predicate, Duration timeout) {
        long limit = timeout.toMillis();
        long start = System.currentTimeMillis();

        int times = 1;
        T result = null;
        while (true) {
            result = supplier.get();
            if (predicate.test(result)) {
                break;
            }
            if (System.currentTimeMillis() - start > limit) {
                break;
            }

            try {
                TimeUnit.SECONDS.sleep(times);
            } catch (InterruptedException ignore) {
            }
            times++;
        }
        return result;
    }

    @AllArgsConstructor
    @Data
    static class AnalyseResult {
        /**
         * 总记录数
         */
        private Long total;

        /**
         * 是否同步
         */
        private Boolean sync;
    }
}
