package com.elitescloud.boot.excel.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.elitescloud.boot.excel.common.param.AbstractExportQueryParam;
import com.elitescloud.boot.excel.config.ExportExcelProperties;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.param.ExportColumnParam;
import com.elitescloud.cloudt.core.service.ExportExcelService;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.StopWatch;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/09/18
 */
@Slf4j
public class ExportExcelServiceImpl<T extends AbstractExportQueryParam, R extends Serializable, E extends Serializable> implements ExportExcelService<T, R, E> {

    private final ExportExcelProperties exportExcelProperties;
    private final TaskExecutor taskExecutor;

    private static final ObjectMapper OBJECT_MAPPER;
    private final DateTimeFormatter formatterLong = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");

    static {
        var pattern = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        OBJECT_MAPPER = new Jackson2ObjectMapperBuilder()
                .serializerByType(LocalDateTime.class, new LocalDateTimeSerializer(pattern))
                .deserializerByType(LocalDateTime.class, new LocalDateTimeDeserializer(pattern))
                .failOnUnknownProperties(false)
                .failOnEmptyBeans(false)
                .build();
    }

    public ExportExcelServiceImpl(ExportExcelProperties exportExcelProperties, TaskExecutor taskExecutor) {
        this.exportExcelProperties = exportExcelProperties;
        this.taskExecutor = taskExecutor;
    }

    @Override
    public Boolean exportSync(T queryParam, HttpServletResponse response, Function<T, PagingVO<R>> dataService) {
        try {
            return this.export(queryParam, response, dataService).get();
        } catch (Exception e) {
            throw new BusinessException("导出失败", e);
        }
    }

    @Override
    public CompletableFuture<Boolean> export(T queryParam, HttpServletResponse response, Function<T, PagingVO<R>> dataService) {
        return export(queryParam, response, dataService, null);
    }

    @Override
    public Boolean exportSync(T queryParam, HttpServletResponse response, Function<T, PagingVO<R>> dataService, Function<R, List<E>> detailData) {
        try {
            return this.export(queryParam, response, dataService, detailData).get();
        } catch (Exception e) {
            throw new BusinessException("导出失败", e);
        }
    }

    @Override
    public CompletableFuture<Boolean> export(T queryParam, HttpServletResponse response, Function<T, PagingVO<R>> dataService, Function<R, List<E>> detailData) {
        return CompletableFuture.supplyAsync(() -> execute(queryParam, response, dataService, detailData), taskExecutor)
                .exceptionally(e -> {
                    log.error("导出excel文件失败", e);

                    response.reset();
                    response.setContentType(MediaType.APPLICATION_JSON_VALUE);
                    response.setCharacterEncoding(StandardCharsets.UTF_8.name());
                    try {
                        response.getWriter().println(OBJECT_MAPPER.writeValueAsString(ApiResult.fail("导出文件失败")));
                    } catch (IOException ex) {
                        ex.printStackTrace();
                    }
                    return true;
                });
    }

    private boolean execute(T queryParam, HttpServletResponse response, Function<T, PagingVO<R>> dataService, Function<R, List<E>> detailData) {
        log.info("开始导出数据...");
        var watch = new StopWatch();
        watch.start();
        checkBeforeExport(queryParam, detailData);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        var fileName = CharSequenceUtil.blankToDefault(queryParam.getFileName(), formatterLong.format(LocalDateTime.now()) + ".xlsx");
        response.setHeader(HttpHeaders.CONTENT_DISPOSITION, ContentDisposition.builder("attachment").filename(fileName, StandardCharsets.UTF_8).build().toString());

        ServletOutputStream outputStream = null;
        try {
            outputStream = response.getOutputStream();
        } catch (IOException e) {
            throw new BusinessException("导出数据异常");
        }

        var excelWriter = EasyExcelFactory.write(outputStream).build();

        StopWatch watchData = null;
        try {
            watchData = writeData(excelWriter, queryParam, dataService, detailData);
        } finally {
            excelWriter.finish();
        }

        watch.stop();
        log.info("导出数据结束，共用时{}s，其中查询业务数据共用时{}s！", watch.getTotalTimeSeconds(), watchData.getTotalTimeSeconds());
        return true;
    }

    private void checkBeforeExport(T queryParam, Function<R, List<E>> detailData) {
        if (queryParam == null) {
            throw new BusinessException("未知导出列");
        }

        // 默认第一页
        queryParam.setCurrent(NumberUtil.max(1, ObjectUtil.defaultIfNull(queryParam.getCurrent(), 1)));
        queryParam.setSize(NumberUtil.max(50, ObjectUtil.defaultIfNull(queryParam.getSize(), 10)));

        if (CollUtil.isEmpty(queryParam.getExportColumn())) {
            throw new BusinessException("未知导出列");
        }

        if (CollUtil.isNotEmpty(queryParam.getExportDetailColumn()) && detailData == null) {
            throw new BusinessException("无法获取详细数据");
        }
    }

    private StopWatch writeData(ExcelWriter excelWriter, T queryParam, Function<T, PagingVO<R>> dataService, Function<R, List<E>> detailData) {
        var sheetBuilder = new ExcelWriterSheetBuilder(excelWriter);
        sheetBuilder.sheetName("导出数据");

        // 导出数据量的限制
        var limit = obtainLimitSize();

        // 转换标题信息
        var headersMain = convertHeaders(queryParam.getExportColumn());
        var headersDetail = convertHeaders(queryParam.getExportDetailColumn());
        var hasDetail = !headersDetail.isEmpty();

        var titles = headersMain.values();
        if (hasDetail) {
            titles = new ArrayList<>(titles);
            titles.addAll(headersDetail.values());
        }
        var head = titles.stream().map(List::of).collect(Collectors.toList());
        sheetBuilder.head(head);
        var sheet = sheetBuilder.build();

        // 查询数据所用时间
        var watch = new StopWatch();
        watch.start();

        // 查询数据
        var data = dataService.apply(queryParam);
        watch.stop();

        var fieldsMain = headersMain.keySet();
        var fieldsDetail = headersDetail.keySet();
        List<List<String>> records = null;
        var page = queryParam.getCurrent() + 1;
        long num = 0;
        while (data != null && CollUtil.isNotEmpty(data.getRecords())) {
            records = convertRecords(data.getRecords(), fieldsMain, fieldsDetail, detailData, limit, num);
            num += records.size();
            excelWriter.write(records, sheet);

            if (limit != null && num >= limit) {
                // 已超出限制
                break;
            }
            if (queryParam.getSize() > data.getRecords().size() || num == data.getTotal()) {
                // 已无数据
                break;
            }
            // 查询下一页数据
            queryParam.setCurrent(++page);
            watch.start();
            data = dataService.apply(queryParam);
            watch.stop();
        }

        if (num == 0) {
            // 没有数据时，写入一空行
            excelWriter.write(null, sheet);
        }

        return watch;
    }

    private Long obtainLimitSize() {
        var limit = exportExcelProperties.getExportLimit();
        return limit == null || limit < 1 ? null : limit;
    }

    private Map<String, String> convertHeaders(List<ExportColumnParam> columnParams) {
        if (CollUtil.isEmpty(columnParams)) {
            return Collections.emptyMap();
        }

        Map<String, String> headerMap = new LinkedHashMap<>(columnParams.size());
        for (var column : columnParams) {
            if (CharSequenceUtil.isNotBlank(column.getField())) {
                headerMap.put(column.getField(), column.getTitle());
            }
        }
        return headerMap;
    }

    private List<List<String>> convertRecords(List<R> dataList, Set<String> fieldsMain, Set<String> fieldsDetail,
                                              Function<R, List<E>> detailData, Long limit, Long num) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }

        if (limit != null && num + dataList.size() > limit) {
            dataList = CollUtil.sub(dataList, 0, (int) (limit - num));
        }

        if (detailData == null) {
            // 无明细的，直接list全部转换
            return OBJECT_MAPPER.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
            }).stream().map(data -> fieldsMain.stream().map(d -> obtainValue(data, d)).collect(Collectors.toList())
            ).collect(Collectors.toList());
        }

        List<List<String>> valueList = new ArrayList<>(512);
        for (var data : dataList) {
            Map<String, Object> dataMap = OBJECT_MAPPER.convertValue(data, new TypeReference<>() {
            });
            var tempValueList = fieldsMain.stream().map(d -> obtainValue(dataMap, d)).collect(Collectors.toList());
            var detailList = detailData.apply(data);
            if (CollUtil.isEmpty(detailList)) {
                // 没有明细数据
                valueList.add(tempValueList);
                continue;
            }

            // 转换明细记录
            var tempDetailValueList = OBJECT_MAPPER.convertValue(detailList, new TypeReference<List<Map<String, Object>>>() {
                    })
                    .stream()
                    .map(detail -> {
                        var temp = new ArrayList<>(tempValueList);
                        temp.addAll(fieldsDetail.stream().map(d -> obtainValue(detail, d)).collect(Collectors.toList()));
                        return temp;
                    })
                    .collect(Collectors.toList());
            valueList.addAll(tempDetailValueList);
        }

        return valueList;
    }

    private String obtainValue(Map<String, Object> map, String key) {
        var val = map.get(key);
        return val == null ? null : val.toString();
    }
}
