package com.elitescloud.boot.excel.config.tmpl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.poi.excel.RowUtil;
import cn.hutool.poi.excel.cell.CellUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.metadata.CellData;
import com.alibaba.excel.metadata.Head;
import com.alibaba.excel.write.handler.AbstractCellWriteHandler;
import com.alibaba.excel.write.metadata.holder.WriteSheetHolder;
import com.alibaba.excel.write.metadata.holder.WriteTableHolder;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.excel.common.DataExport;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.excel.common.param.ImportRecordRespVO;
import com.elitescloud.boot.excel.config.tmpl.export.ExportStrategyParam;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.boot.excel.config.tmpl.export.strategy.TmplExportStrategyDelegate;
import com.elitescloud.boot.excel.util.ExcelImportUtil;
import com.elitescloud.boot.excel.util.ExcelUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.AbstractOrderQueryParam;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.DatetimeUtil;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import com.elitescloud.cloudt.system.dto.resp.ExportResultRespVO;
import com.elitescloud.cloudt.system.dto.resp.ImportRateRespVO;
import com.elitescloud.cloudt.system.dto.resp.ImportResultRespVO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.log4j.Log4j2;
import org.apache.poi.ss.usermodel.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.task.TaskExecutor;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.*;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 文件导入导出服务.
 *
 * @author Kaiser（wang shao）
 * @date 2/17/2023
 */
@Log4j2
class TmplDataService implements ApplicationRunner {

    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private ObjectMapper objectMapper;
    @Autowired
    private DataImportServiceFactory dataImportServiceFactory;
    @Autowired
    private DataExportServiceFactory dataExportServiceFactory;
    @Autowired
    private TmplExportStrategyDelegate exportStrategyDelegate;
    private final FileService<?> fileService;

    private final SystemTmplDataSupport fsmTmplSupport;

    // 临时文件夹
    private File tempDir = null;

    public TmplDataService(FileService<?> fileService, SystemTmplDataSupport fsmTmplSupport) {
        this.fileService = fileService;
        this.fsmTmplSupport = fsmTmplSupport;
        // 初始化临时文件夹
        initTempDir();
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        // 处理系统异常停止的记录状态
        CompletableFuture.runAsync(() -> {
            fsmTmplSupport.updateResultForSysError("服务停止");
        }, taskExecutor).exceptionally(throwable -> {
            log.warn("处理未导入结束的失败", throwable);
            return null;
        });
    }

    private void initTempDir() {
        tempDir = new File(System.getProperty("java.io.tmpdir"));
        if (tempDir.exists()) {
            return;
        }
        if (!tempDir.mkdirs()) {
            throw new IllegalArgumentException("创建临时文件夹失败：" + tempDir.getAbsolutePath());
        }
    }

    /**
     * 根据模板编号下载模板文件
     *
     * @param code 模板编号
     * @return 模板文件
     */
    public HttpEntity<StreamingResponseBody> downloadByCode(@NotBlank String code) {
        return fsmTmplSupport.downloadByCode(code);
    }

    /**
     * 导入数据
     *
     * @param code     模板编号
     * @param dataFile 数据文件
     * @return 导入结果
     */
    public ApiResult<ImportResultRespVO> importData(@NotBlank String code, @NotNull MultipartFile dataFile) {
        // 导入前处理
        var prepareResult = this.prepareForDeal(code, null, dataFile);
        var tmplDTO = prepareResult.getTmplDTO();
        var recordId = prepareResult.getRecordId();

        // 获取数据导入服务
        var dataImportService = dataImportServiceFactory.getDataImportService(code);

        // 开始导入
        ImportResultRespVO respVO = null;
        try {
            respVO = this.startImport(recordId, tmplDTO, dataFile, dataImportService);
        } catch (Exception exception) {
            log.error("导入失败", exception);
            this.afterImport(tmplDTO, recordId, 0L, ExceptionsUtil.getCauseMsg(exception));
            return ApiResult.fail(ExceptionsUtil.normalize(exception, "导入失败").getMessage());
        }
        if (Boolean.TRUE.equals(respVO.getSync())) {
            this.afterImport(tmplDTO, recordId, (long) ObjectUtil.defaultIfNull(respVO.getSyncResult().getNumSuccess(), 0), null);
        }
        return ApiResult.ok(respVO);
    }

    /**
     * 导出数据
     *
     * @param code       模板编码
     * @param queryParam 查询参数
     * @return 导出结果
     */
    public ApiResult<ExportResultRespVO> exportData(@NotBlank String code, Map<String, Object> queryParam) {
        // 导出前处理
        var prepareResult = this.prepareForDeal(code, queryParam, null);
        var tmplDTO = prepareResult.getTmplDTO();
        var recordId = prepareResult.getRecordId();

        // 获取数据导出服务
        var dataExportService = dataExportServiceFactory.getDataExportService(code);

        // 开始导出
        AnalyseResult result;
        try {
            result = startExport(recordId, tmplDTO, queryParam, dataExportService);
        } catch (Exception exception) {
            log.error("导出失败", exception);
            afterImport(tmplDTO, recordId, 0L, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导出失败" + (exception instanceof BusinessException ? "，" + exception.getMessage() : ""));
        }

        return ApiResult.ok(new ExportResultRespVO(result.getSync(), recordId, result.getTotal()));
    }

    /**
     * 下載导出文件
     *
     * @param id 记录ID
     * @return 文件
     */
    public HttpEntity<StreamingResponseBody> downloadExportFile(Long id) {
        AtomicInteger times = new AtomicInteger(1);
        String code = TmplDataService.retry(() -> fsmTmplSupport.getRecordFileCode(id), c -> {
            if (CharSequenceUtil.isNotBlank(c)) {
                return true;
            }
            if (times.getAndAdd(1) >= 5) {
                return true;
            }
            return false;
        }, Duration.ofMinutes(1));
        if (CharSequenceUtil.isBlank(code)) {
            log.error("下载导出记录{}的文件失败，文件不存在", id);
            return ResponseEntity.badRequest().build();
        }

        return fileService.download(code, null);
    }

    public ApiResult<ImportRateRespVO> getRate(Long id) {
        SysImportRateDTO rateDTO = null;
        try {
            rateDTO = fsmTmplSupport.getImportRateFromCache(id);
            if (rateDTO == null) {
                rateDTO = fsmTmplSupport.getImportRate(id);
            }
        } catch (Exception exception) {
            log.error("查询进度失败", exception);
            return ApiResult.fail("查询进度失败");
        }

        if (rateDTO == null) {
            return ApiResult.fail("记录不存在");
        }
        ImportRateRespVO respVO = new ImportRateRespVO();
        respVO.setFinish(rateDTO.getFinish());
        respVO.setTotal(rateDTO.getTotal());

        if (Boolean.TRUE.equals(rateDTO.getFinish()) || ObjectUtil.equals(rateDTO.getTotal(), rateDTO.getCount())) {
            respVO.setCount(rateDTO.getTotal());
            respVO.setFinish(true);
            respVO.setRate("100%");
        } else {
            long count = ObjectUtil.defaultIfNull(rateDTO.getCount(), 0).longValue();
            long total = ObjectUtil.defaultIfNull(rateDTO.getTotal(), 1).longValue();
            respVO.setCount(rateDTO.getCount());
            respVO.setRate(BigDecimal.valueOf(count * 1.0 / total * 100).setScale(0, RoundingMode.DOWN) + "%");
        }
        respVO.setNumSuccess(rateDTO.getNumSuccess());
        respVO.setFailFileCode(rateDTO.getFailFileCode());

        return ApiResult.ok(respVO);
    }

    /**
     * 查询导入导出记录
     *
     * @param tmplCode 模板编码
     * @param hours    小时数
     * @param self     是否只查自己的
     * @return 记录列表
     */
    public ApiResult<List<ImportRecordRespVO>> queryRecord(@NotBlank String tmplCode, Integer hours, Boolean self) {
        var recordDtoList = fsmTmplSupport.queryRecord(tmplCode, hours, self);
        if (CollUtil.isEmpty(recordDtoList)) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询文件信息
        var fileCodes = recordDtoList.stream().flatMap(t -> Stream.of(t.getFileCode(), t.getFailFileCode())).filter(StringUtils::hasText).collect(Collectors.toList());
        Map<String, FileObjRespVO<?>> fileInfoMap = null;
        if (CollUtil.isNotEmpty(fileCodes)) {
            var fileInfoResult = fileService.query(fileCodes);
            if (!fileInfoResult.isSuccess()) {
                log.error("查询文件异常：{}", fileInfoResult.getMsg());
            }
            if (CollUtil.isNotEmpty(fileInfoResult.getData())) {
                fileInfoMap = fileInfoResult.getData().stream().collect(Collectors.toMap(FileObjRespVO::getFileCode, t -> t, (t1, t2) -> t1));
            }
        }
        fileInfoMap = fileInfoMap == null ? Collections.emptyMap() : fileInfoMap;

        Map<String, FileObjRespVO<?>> finalFileInfoMap = fileInfoMap;
        var respVoList = recordDtoList.stream().map(t -> {
            ImportRecordRespVO respVO = new ImportRecordRespVO();
            respVO.setId(t.getId());
            respVO.setUserName(t.getUserName());
            respVO.setFileInfo(t.getFileCode() == null ? null : finalFileInfoMap.get(t.getFileCode()));
            respVO.setTimeImport(t.getTimeImport());
            respVO.setTimeFinish(t.getTimeFinish());
            respVO.setFinish(t.getFinish());
            respVO.setNumTotal(t.getNumTotal());
            respVO.setNumSuc(t.getNumSuc());
            respVO.setFailFileInfo(t.getFailFileCode() == null ? null : finalFileInfoMap.get(t.getFailFileCode()));
            respVO.setFailReason(t.getFailReason());

            return respVO;
        }).collect(Collectors.toList());
        return ApiResult.ok(respVoList);
    }

    private PreparationResult prepareForDeal(@NotBlank String code, Map<String, Object> queryParam, MultipartFile importFile) {
        // 获取模板信息
        var tmplDTO = fsmTmplSupport.getTmplByCode(code);

        // 执行前的校验
        String msg = this.validateBeforeDeal(tmplDTO, queryParam, importFile);
        Assert.isTrue(CharSequenceUtil.isBlank(msg), msg);

        // 保存记录
        Long recordId = null;
        try {
            recordId = saveRecord(tmplDTO, importFile, queryParam);
        } catch (Exception exception) {
            boolean isExport = Boolean.TRUE.equals(tmplDTO.getExport());
            throw ExceptionsUtil.normalize(exception, isExport ? "导出数据异常" : "导入数据异常");
        }

        return new PreparationResult(tmplDTO, recordId);
    }

    private String validateBeforeDeal(SysTmplDTO tmplDTO, Map<String, Object> queryParam, MultipartFile importFile) {
        if (tmplDTO == null || Boolean.FALSE.equals(tmplDTO.getEnabled())) {
            return "模板不存在或已禁用";
        }
        if (CollUtil.isEmpty(tmplDTO.getAttributeFields())) {
            return "模板无效，未发现数据字段，请联系管理员重新配置";
        }

        boolean export = Boolean.TRUE.equals(tmplDTO.getExport());
        if (export) {
            if (!dataExportServiceFactory.isSupport(tmplDTO.getCode())) {
                return "未发现有效的数据导出服务";
            }
        } else {
            Assert.notNull(importFile, "根据导入模板" + tmplDTO.getCode() + "导入数据时未发现有效导入文件");
            if (!dataImportServiceFactory.isSupport(tmplDTO.getCode())) {
                return "未发现有效的数据导入服务";
            }
        }

        // 限流设置
        if (!fsmTmplSupport.updateLimiter(tmplDTO, true)) {
            return "当前访问用户过多，请稍后再试";
        }

        return null;
    }


    private Long saveRecord(SysTmplDTO tmplDTO, MultipartFile dataFile, Map<String, Object> queryParam) {
        return fsmTmplSupport.saveRecord(tmplDTO.getCode(), dataFile, queryParam);
    }

    private void afterImport(SysTmplDTO tmplDTO, Long recordId, Long numSuc, String msg) {
        this.afterImport(tmplDTO, recordId, numSuc, msg, null);
    }

    private void afterImport(SysTmplDTO tmplDTO, Long recordId, Long numSuc, String msg, File file) {
        // 更新导入结果
        if (recordId != null) {
            String fileCode = uploadImportFile(file);
            var importResultDTO = RecordResultSaveDTO.builder()
                    .recordId(recordId)
                    .success(CharSequenceUtil.isBlank(msg))
                    .numSuc(numSuc)
                    .failMsg(msg)
                    .fileCode(fileCode)
                    .build();
            fsmTmplSupport.updateImportResult(importResultDTO);
        } else {
            log.error("更新导入导出记录结果失败：{}", msg);
        }

        // 更新限流信息
        fsmTmplSupport.updateLimiter(tmplDTO, false);
    }

    private String uploadImportFile(File file) {
        if (file == null) {
            return null;
        }
        var uploadResult = fileService.upload(file);

        // 删除临时文件
        file.delete();
        if (uploadResult.isSuccess() && uploadResult.getData() != null) {
            return uploadResult.getData().getFileCode();
        }
        log.error("上传导入导出结果文件失败：{}", uploadResult);
        return null;
    }

    private List<?> analyseData(SysTmplDTO tmplDTO, MultipartFile dataFile, DataImportServiceFactory.ServiceMetaData dataImportService) {
        try {
            return ExcelImportUtil.instance(dataFile.getInputStream())
                    .headRow(tmplDTO.getHeadRow())
                    .dataType(dataImportService.getDataType(), tmplDTO.getAttributeFields())
                    .readAllSync();
        } catch (Exception e) {
            throw new BusinessException("解析导入数据失败", e);
        }
    }

    private AnalyseResult startExport(Long recordId, SysTmplDTO tmplDTO, Map<String, Object> queryParam, DataExportServiceFactory.ServiceMetaData dataExportService) {
        var exportService = dataExportService.getDataExport();
        var param = convertParam(queryParam, dataExportService.getParamType());
        int pageSize = obtainPageSize(dataExportService.getDataExport());
        param.setCurrent(1);
        param.setSize(pageSize);
        var queryResult = exportService.executeExport(param);
        if (queryResult == null || queryResult.getTotal() == 0) {
            throw new BusinessException("没有符合条件的数据");
        }

        int limit = ObjectUtil.defaultIfNull(tmplDTO.getDataLimitPer(), -1);
        if (limit != -1 && queryResult.getTotal() > limit) {
            throw new BusinessException("每次最多允许导出" + limit + "条");
        }
        // 更新导出总数量
        CompletableFuture.runAsync(() -> fsmTmplSupport.updateImportNum(recordId, queryResult.getTotal()), taskExecutor);

        // 判断是否需要异步
        var sync = tmplDTO.getAsyncThreshold() == null || tmplDTO.getAsyncThreshold() == -1 || queryResult.getTotal() <= tmplDTO.getAsyncThreshold();

        if (sync) {
            exportSync(recordId, tmplDTO, queryResult, param, exportService);

            return new AnalyseResult(queryResult.getTotal(), true);
        }
        exportAsync(recordId, tmplDTO, queryResult, param, exportService);

        return new AnalyseResult(queryResult.getTotal(), false);
    }

    private void exportAsync(Long recordId, SysTmplDTO tmplDTO, PagingVO<Serializable> pageData, AbstractOrderQueryParam queryParam,
                             DataExport<Serializable, AbstractOrderQueryParam> dataExport) {
        // 设置当前的用户认证
        var authentication = SecurityContextHolder.getContext().getAuthentication();

        CompletableFuture.supplyAsync(() -> {
                    if (authentication != null) {
                        SecurityContextHolder.getContext().setAuthentication(authentication);
                    }
                    return write2Excel(false, recordId, tmplDTO, pageData, queryParam, dataExport);
                }, taskExecutor)
                .whenComplete((ret, throwable) -> {
                    fsmTmplSupport.removeRate(recordId);
                    if (throwable == null) {
                        afterImport(tmplDTO, recordId, ret, null);
                    } else {
                        afterImport(tmplDTO, recordId, 0L, ExceptionUtil.getRootCause(throwable).getMessage());
                        log.error("导出数据时出现异常：", throwable);
                    }
                });
    }

    private int obtainPageSize(DataExport<Serializable, AbstractOrderQueryParam> dataExportService) {
        Integer pageSize = dataExportService.pageSize();
        if (pageSize == null || pageSize < 1 || pageSize > 1000) {
            // 默认页大小
            return 500;
        }

        return pageSize;
    }

    private int obtainStepSize(DataImport<Serializable> dataImport) {
        Integer stepSize = dataImport.stepSize();
        if (stepSize == null || stepSize < 1 || stepSize > 1000) {
            // 默认页大小
            return 10;
        }

        return stepSize;
    }

    private void exportSync(Long recordId, SysTmplDTO tmplDTO,
                            PagingVO<Serializable> pageData, AbstractOrderQueryParam queryParam, DataExport<Serializable, AbstractOrderQueryParam> dataExport) {
        Long num = write2Excel(true, recordId, tmplDTO, pageData, queryParam, dataExport);
        afterImport(tmplDTO, recordId, num, null);
    }

    private Long write2Excel(boolean sync, Long recordId, SysTmplDTO tmplDTO, PagingVO<Serializable> pageData,
                             AbstractOrderQueryParam queryParam, DataExport<Serializable, AbstractOrderQueryParam> dataExport) {
        var param = new ExportStrategyParam()
                .setSync(sync)
                .setImportId(recordId)
                .setTmplDTO(tmplDTO)
                .setFirstPageData(pageData)
                .setQueryParam(queryParam)
                .setDataExport(dataExport);
        return exportStrategyDelegate.export(param);
    }

    private <E extends Serializable> E convertParam(Map<String, Object> queryParam, Class<E> paramType) {
        try {
            return objectMapper.convertValue(queryParam, paramType);
        } catch (IllegalArgumentException e) {
            throw new BusinessException("转换查询参数失败，请检查参数格式", e);
        }
    }

    private ImportResultRespVO startImport(Long recordId, SysTmplDTO tmplDTO, MultipartFile dataFile,
                                           DataImportServiceFactory.ServiceMetaData dataImportService) {
        // 判断是否需要异步
        List dataList = this.analyseData(tmplDTO, dataFile, dataImportService);
        var total = dataList.size();

        if (total == 0) {
            throw new BusinessException("导入数据为空");
        }
        if (tmplDTO.getDataLimitPer() != -1 && total > tmplDTO.getDataLimitPer()) {
            throw new BusinessException("每次最多允许导入" + tmplDTO.getDataLimitPer() + "条");
        }

        // 更新导入总数量
        fsmTmplSupport.updateImportNum(recordId, (long) total);
        // 复制临时文件，避免异步时删掉
        File tempDataFile = FileUtil.createTempFile();
        String dataFileName = dataFile.getOriginalFilename();
        try {
            dataFile.transferTo(tempDataFile);
        } catch (IOException e) {
            throw new IllegalStateException("创建临时文件异常", e);
        }

        var sync = tmplDTO.getAsyncThreshold() == null || tmplDTO.getAsyncThreshold() == -1 || total <= tmplDTO.getAsyncThreshold();
        if (sync) {
            // 需要同步处理
            var syncResult = this.executeImport(dataImportService.getDataImport(), dataList, tmplDTO.getHeadRow() + 1);
            // 保存失败记录
            var failFileCode = this.saveFailRecord(recordId, tempDataFile, dataFileName, tmplDTO, dataList, syncResult.getFailRecords());
            syncResult.setFailFileCode(failFileCode);

            return ImportResultRespVO.builder().sync(true).syncResult(syncResult).build();
        }

        // 需要异步处理
        CompletableFuture.supplyAsync(() -> this.importAsync(tmplDTO, recordId, dataImportService.getDataImport(), dataList), taskExecutor)
                .whenComplete((ret, throwable) -> {
                    // 保存失败记录
                    if (ret != null) {
                        this.saveFailRecord(recordId, tempDataFile, dataFileName, tmplDTO, dataList, ret.getFailRecords());
                    }

                    // 更新导入结果
                    int numSuccess = ret == null ? 0 : ret.getNumSuccess();
                    if (throwable == null) {
                        // 导入成功
                        afterImport(tmplDTO, recordId, (long) numSuccess, null);
                    } else {
                        // 导入失败
                        afterImport(tmplDTO, recordId, (long) numSuccess, ObjectUtil.defaultIfNull(ExceptionUtil.getRootCause(throwable), throwable).getMessage());
                        log.error("导入数据时出现异常：", throwable);
                    }
                    // 删掉导入进度
                    fsmTmplSupport.removeRate(recordId);
                    // 删除临时文件
                    tempDataFile.delete();
                });
        return ImportResultRespVO.builder().sync(false).asyncResult(ImportResultRespVO.AsyncResult.builder().importId(recordId).build()).build();
    }

    private ImportResultRespVO.SyncResult importAsync(SysTmplDTO tmplDTO, Long recordId, DataImport<Serializable> dataImportService, List<Serializable> dataList) {
        int total = dataList.size();
        var size = this.obtainStepSize(dataImportService);
        var start = 0;
        var count = 0;
        var numSuc = 0;
        ImportResultRespVO.SyncResult tempResult = null;
        List<String> fail = new ArrayList<>(1024);
        for (var i = 0; ; i++) {
            start = i * size;
            count = Math.min(start + size, total);
            List<Serializable> toImportData = dataList.subList(start, count);
            tempResult = this.executeImport(dataImportService, toImportData, tmplDTO.getHeadRow() + 1 + start);

            numSuc += tempResult.getNumSuccess();
            fail.addAll(tempResult.getFailRecords());

            var finish = count >= total;
            fsmTmplSupport.storeRate(recordId, SysImportRateDTO.builder()
                    .total((long) total)
                    .finish(false)
                    // 减去1，用来解决错误文件尚未处理成功
                    .count((long) count - 1)
                    .numSuccess((long) numSuc)
                    .tmplCode(dataImportService.getTmplCode())
                    .build());
            if (finish) {
                break;
            }
        }

        return ImportResultRespVO.SyncResult.builder().total(total).numSuccess(numSuc).failRecords(fail).build();
    }

    private ImportResultRespVO.SyncResult executeImport(DataImport<Serializable> dataImportService, List<Serializable> dataList, int startIndex) {
        var total = dataList.size();
        List<String> failRecords = null;
        try {
            failRecords = dataImportService.executeImport(dataList, startIndex);
            if (failRecords == null) {
                // 兼容老的
                var tempResult = dataImportService.execute(dataList, startIndex);
                failRecords = tempResult == null ? null : tempResult.getFailRecords();
            }
        } catch (Throwable e) {
            log.error("导入失败", e);
            failRecords = CollectionUtil.toList(ObjectUtil.defaultIfNull(ExceptionsUtil.getRootCause(e), e).getMessage(), total);
        }
        if (failRecords == null) {
            // 默认全成功
            failRecords = CollectionUtil.toList(null, total);
        } else if (failRecords.size() != total) {
            log.error(dataImportService.getTmplCode() + "返回的导入结果与数据量不匹配");
        }

        return ImportResultRespVO.SyncResult.builder()
                .total(total)
                .numSuccess((int) failRecords.stream().filter(CharSequenceUtil::isBlank).count())
                .failRecords(failRecords)
                .build();
    }

    private String saveFailRecord(long importId, File dataFile, String dataFileName, SysTmplDTO tmplDTO, List dataList, List<String> failRecords) {
        boolean success = failRecords == null || failRecords.isEmpty() || failRecords.stream().noneMatch(StringUtils::hasText);
        if (success) {
            log.info("导入成功，无需记录失败文件！");
            return null;
        }
        if (dataList.size() != failRecords.size()) {
            log.error(tmplDTO.getCode() + "数据数量与失败记录数量不一致，无法保存失败记录");
            return null;
        }

        // 转换结果数据
        List<List<String>> failData = null;
        try {
            failData = this.convertFailData(tmplDTO, dataList, failRecords);
        } catch (Exception e) {
            log.error("转换导入失败的数据异常：", e);
            return null;
        }

        // 写入文件
        try {
            var failRecordFile = this.writeFailRecordToFile(dataFile, dataFileName, tmplDTO, failData);
            // 上传文件
            var fileUploadResult = fileService.upload(failRecordFile);
            failRecordFile.delete();
            if (fileUploadResult.getData() == null) {
                log.error("上传失败记录文件失败：{}", fileUploadResult.getMsg());
                return null;
            }
            // 更新失败文件记录
            fsmTmplSupport.saveImportFailRecord(importId, fileUploadResult.getData().getFileCode());
            return fileUploadResult.getData().getFileCode();
        } catch (Exception e) {
            log.error("保存失败记录异常：", e);
        }
        return null;
    }

    private File writeFailRecordToFile(File dataFile, String dataFileName, SysTmplDTO tmplDTO, List<List<String>> failRecords) throws Exception {
        // 创建临时文件
        var fileName = CharSequenceUtil.blankToDefault(dataFileName, dataFile.getName());
        var indexDot = fileName.indexOf(".");
        fileName = indexDot > 0 ? fileName.substring(0, indexDot) : "";
        fileName = "Error_" + fileName + "_" + DatetimeUtil.FORMATTER_DATETIME_LONG.format(LocalDateTime.now()) + ".xlsx";
        var tempDir = new File(System.getProperty("java.io.tmpdir"));
        if (!tempDir.exists() && !tempDir.mkdirs()) {
            throw new IllegalStateException("创建临时文件" + System.getProperty("java.io.tmpdir") + "失败");
        }
        var failFile = new File(tempDir, fileName);
        failFile.createNewFile();

        // 写入失败记录
        var colSize = failRecords.get(0).size();
        EasyExcel.write(failFile)
                .relativeHeadRowIndex(tmplDTO.getHeadRow())
                .sheet("Sheet1")
                .registerWriteHandler(new FailDataExcelCellWriteHandler(colSize - 1))
                .doWrite(failRecords);

        // 写入头部信息
        var sheetOriginal = ExcelUtil.getReader(dataFile, 0).getSheet();
        try (
                FileInputStream failFileInputStream = new FileInputStream(failFile);
        ) {
            var sheetFail = cn.hutool.poi.excel.ExcelUtil.getReader(failFileInputStream).getSheet();
            writeFailTitle(sheetFail, colSize - 1);
            ExcelUtil.copyRows(sheetOriginal, sheetFail, 0, 1);
            try (FileOutputStream failFileOutStream = new FileOutputStream(failFile)) {
                sheetFail.getWorkbook().write(failFileOutStream);
                sheetFail.getWorkbook().close();
            }
        }

        return failFile;
    }

    private void writeFailTitle(Sheet sheet, int msgColIndex) {
        // 设置列宽，10个汉字
        sheet.setColumnWidth(msgColIndex, 20 * 256);

        var row = RowUtil.getOrCreateRow(sheet, 0);
        var col = CellUtil.getOrCreateCell(row, msgColIndex);

        var style = sheet.getWorkbook().createCellStyle();
        style.cloneStyleFrom(col.getCellStyle());
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setAlignment(HorizontalAlignment.CENTER);
        var font = sheet.getWorkbook().createFont();
        font.setColor(Font.COLOR_RED);
        font.setBold(true);
        style.setFont(font);

        col.setCellStyle(style);

        col.setCellValue("错误信息");
    }

    private List<List<String>> convertFailData(SysTmplDTO tmplDTO, List dataList, List<String> failRecords) throws Exception {
        List<List<String>> recordList = new ArrayList<>(failRecords.size());
        int i = -1;
        Map<String, Object> valueMap = null;
        List<String> record = null;
        for (String failRecord : failRecords) {
            i++;
            if (CharSequenceUtil.isBlank(failRecord)) {
                continue;
            }
            // 转换数据至map
            valueMap = objectMapper.readValue(objectMapper.writeValueAsString(dataList.get(i)), new TypeReference<>() {
            });
            record = new ArrayList<>(valueMap.size());
            Object value = null;
            for (String attributeField : tmplDTO.getAttributeFields()) {
                value = ObjectUtil.defaultIfNull(valueMap.get(attributeField), "");
                if (value instanceof String) {
                    record.add((String) value);
                } else {
                    record.add(objectMapper.writeValueAsString(value));
                }
            }
            // 加入失败原因
            record.add(failRecord);
            recordList.add(record);
        }
        return recordList;
    }

    /**
     * 重试
     *
     * @param supplier  功能
     * @param predicate 判断是否满足
     * @param timeout   超时时间
     * @param <T>       结果类型
     * @return 结果
     */
    private static <T> T retry(Supplier<T> supplier, Predicate<T> predicate, Duration timeout) {
        long limit = timeout.toMillis();
        long start = System.currentTimeMillis();

        int times = 1;
        T result = null;
        while (true) {
            result = supplier.get();
            if (predicate.test(result)) {
                break;
            }
            if (System.currentTimeMillis() - start > limit) {
                break;
            }

            try {
                TimeUnit.SECONDS.sleep(times);
            } catch (InterruptedException ignore) {
            }
            times++;
        }
        return result;
    }

    static class FailDataExcelCellWriteHandler extends AbstractCellWriteHandler {

        private final int msgColIndex;

        public FailDataExcelCellWriteHandler(int msgColIndex) {
            this.msgColIndex = msgColIndex;
        }

        @Override
        public void afterCellDataConverted(WriteSheetHolder writeSheetHolder, WriteTableHolder writeTableHolder, CellData cellData, Cell cell, Head head, Integer relativeRowIndex, Boolean isHead) {
            if (cell.getColumnIndex() == msgColIndex) {
                // 最后一列，错误信息
                var style = writeSheetHolder.getSheet().getWorkbook().createCellStyle();
                style.cloneStyleFrom(cell.getCellStyle());
                style.setAlignment(HorizontalAlignment.LEFT);
                style.setVerticalAlignment(VerticalAlignment.CENTER);
                // 自动换行
                style.setWrapText(true);

                var font = writeSheetHolder.getSheet().getWorkbook().createFont();
                font.setColor(Font.COLOR_RED);
                style.setFont(font);
                cell.setCellStyle(style);
            }
        }

        @Override
        public void afterCellDispose(WriteSheetHolder writeSheetHolder, WriteTableHolder writeTableHolder, List<CellData> cellDataList, Cell cell, Head head, Integer relativeRowIndex, Boolean isHead) {
        }
    }

    @AllArgsConstructor
    @Data
    static class AnalyseResult {
        /**
         * 总记录数
         */
        private Long total;

        /**
         * 是否同步
         */
        private Boolean sync;
    }

    @AllArgsConstructor
    @Data
    static class PreparationResult {
        /**
         * 模板信息
         */
        private SysTmplDTO tmplDTO;
        /**
         * 记录ID
         */
        private Long recordId;
    }
}
