package com.elitescloud.boot.excel.config.tmpl.export.strategy;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.excel.common.DataExport;
import com.elitescloud.boot.excel.config.tmpl.export.ExportStrategyParam;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import com.elitescloud.cloudt.common.base.param.AbstractOrderQueryParam;
import com.elitescloud.cloudt.context.util.DatetimeUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/28
 */
abstract class BaseExportStrategy implements ExportStrategy {
    private static final Logger LOG = LoggerFactory.getLogger(BaseExportStrategy.class);

    protected static final Integer SHEET_ROW_MAX = 1000000;
    protected final FileService<?> fileService;
    protected final SystemTmplDataSupport fsmTmplSupport;
    protected final ObjectMapper objectMapper;

    protected BaseExportStrategy(FileService<?> fileService, SystemTmplDataSupport fsmTmplSupport, ObjectMapper objectMapper) {
        this.fileService = fileService;
        this.fsmTmplSupport = fsmTmplSupport;
        this.objectMapper = objectMapper;
    }

    /**
     * 表头
     *
     * @param tmplDTO 模板信息
     * @return 表头
     */
    protected List<List<String>> obtainHeaders(SysTmplDTO tmplDTO) {
        Assert.notEmpty(tmplDTO.getAttributes(), "未获取到模板的头部信息");
        return tmplDTO.getAttributes().subList(0, tmplDTO.getFieldTypeRow() - 1);
    }

    /**
     * 表头
     *
     * @param tmplDTO
     * @return
     */
    protected List<String> obtainAttributes(SysTmplDTO tmplDTO) {
        Assert.notEmpty(tmplDTO.getAttributes(), "未获取到模板的头部信息");
        return tmplDTO.getAttributes().get(tmplDTO.getFieldTypeRow() - 1);
    }

    /**
     * 更新导出进度
     *
     * @param recordId 导入导出进度ID
     * @param rateDTO  进度
     */
    protected void updateRate(long recordId, SysImportRateDTO rateDTO) {
        fsmTmplSupport.storeRate(recordId, rateDTO);
    }

    /**
     * 更新导出完毕状态
     *
     * @param param
     * @param numSuc
     * @param msg
     */
    protected void updateExportFinish(ExportStrategyParam param, Long numSuc, String msg) {
        this.updateExportFinish(param, numSuc, msg, null);
    }

    protected void updateExportFinish(ExportStrategyParam param, Long numSuc, String msg, File file) {
        // 更新导入结果
        if (param.getImportId() != null) {
            String fileCode = uploadImportFile(file);
            var importResultDTO = RecordResultSaveDTO.builder()
                    .recordId(param.getImportId())
                    .success(CharSequenceUtil.isBlank(msg))
                    .numSuc(numSuc)
                    .failMsg(msg)
                    .fileCode(fileCode)
                    .build();
            fsmTmplSupport.updateImportResult(importResultDTO);
        } else {
            LOG.error("更新导入导出记录结果失败：{}", msg);
        }

        // 更新限流信息
        fsmTmplSupport.updateLimiter(param.getTmplDTO(), false);
    }

    /**
     * 上传导出的文件
     *
     * @param param
     * @param file
     * @param fileIndex
     */
    protected void updateExportFile(ExportStrategyParam param, File file, int fileIndex) {
        String fileCode = this.uploadImportFile(file);
        fsmTmplSupport.saveExportFile(param.getImportId(), fileCode, fileIndex);
    }

    /**
     * 将业务返回的数据转为字符串列表
     *
     * @param dataList
     * @param fields
     * @return
     */
    protected List<List<String>> convertExportData(List<Serializable> dataList, List<String> fields) {
        return objectMapper.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
                }).stream()
                .map(data -> fields.stream().map(field -> obtainValue(data, field)).collect(Collectors.toList()))
                .collect(Collectors.toList());
    }

    private String uploadImportFile(File file) {
        if (file != null) {
            var uploadResult = fileService.upload(file);
            // 删除临时文件
            file.delete();
            if (uploadResult.isSuccess()) {
                return uploadResult.getData().getFileCode();
            } else {
                LOG.error("上传导入导出结果文件失败：{}", uploadResult);
            }
        }
        return null;
    }

    private String obtainValue(Map<String, Object> data, String field) {
        if (StrUtil.isBlank(field)) {
            return "";
        }
        Object value = data.get(field);
        if (value == null) {
            return "";
        }
        if (value instanceof LocalDateTime) {
            return DatetimeUtil.FORMATTER_DATETIME.format((LocalDateTime) value);
        } else if (value instanceof LocalDate) {
            return DatetimeUtil.FORMATTER_DATE.format((LocalDate) value);
        } else if (value instanceof Date) {
            return DatetimeUtil.FORMAT_DATETIME.format((Date) value);
        } else if (value instanceof Double) {
            return NumberUtil.decimalFormatMoney((Double) value);
        } else if (value instanceof Float) {
            return NumberUtil.decimalFormatMoney((Float) value);
        }
        return value.toString();
    }

    /**
     * 创建导出临时文件
     *
     * @return
     */
    protected File createExportFile(ExportStrategyParam param) {
        return createExportFile(param, null);
    }

    /**
     * 创建导出临时文件
     *
     * @param param
     * @param order
     * @return
     */
    protected File createExportFile(ExportStrategyParam param, Integer order) {
        String fileName = CharSequenceUtil.blankToDefault(param.getDataExport().exportFileName(), param.getTmplDTO().getName());
        if (order != null) {
            fileName = fileName + "_" + order;
        }

        File file = new File(System.getProperty("java.io.tmpdir"), fileName + ".xlsx");
        file.delete();
        try {
            file.createNewFile();
        } catch (IOException e) {
            throw new RuntimeException("创建临时文件失败", e);
        }
        return file;
    }

    /**
     * 获取sheet的数据量限制
     *
     * @param tmplDTO
     * @return
     */
    protected int sheetLimit(SysTmplDTO tmplDTO) {
        Integer limit = tmplDTO.getExportSheetLimit();
        if (limit == null) {
            return SHEET_ROW_MAX;
        }
        if (limit > 0 && limit <= SHEET_ROW_MAX) {
            return limit;
        }

        throw new IllegalArgumentException(tmplDTO.getCode() + "参数" + limit + "不合法");
    }

    /**
     * 分页大小
     *
     * @param dataExportService
     * @return
     */
    protected Integer obtainPageSize(DataExport<Serializable, AbstractOrderQueryParam> dataExportService) {
        Integer pageSize = dataExportService.pageSize();
        if (pageSize == null || pageSize < 1 || pageSize > 1000) {
            // 默认页大小
            return 500;
        }

        return pageSize;
    }
}
