package com.elitescloud.boot.excel.util;

import cn.hutool.poi.excel.RowUtil;
import cn.hutool.poi.excel.cell.CellUtil;
import cn.hutool.poi.excel.style.StyleUtil;
import org.apache.poi.ss.usermodel.Sheet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;

/**
 * Excel工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/5
 */
public final class ExcelUtil extends cn.hutool.poi.excel.ExcelUtil {
    private static final Logger LOG = LoggerFactory.getLogger(ExcelUtil.class);

    private ExcelUtil() {
    }

    /**
     * 复制行
     * <p>
     * 从startIndex复制到endIndex，包含startIndex和endIndex行，索引按从0开始
     *
     * @param source     源文件
     * @param target     目标文件
     * @param startIndex 开始行的索引，以0开始
     * @param endIndex   结束行的索引
     */
    public static void copyRows(@NonNull Sheet source, @NonNull Sheet target, int startIndex, int endIndex) {
        Assert.isTrue(endIndex > startIndex, "endIndex必须大于startIndex");
        var startRowNum = source.getFirstRowNum();
        var endRowNum = source.getLastRowNum();
        if (startIndex > endRowNum || endIndex < startRowNum) {
            // 没有需要复制的行
            LOG.warn("没有可复制的行数：{}-{}, {}-{}", startIndex, endIndex, startRowNum, endRowNum);
            return;
        }

        // 开始复制
        for (int i = startIndex; i <= endIndex; i++) {
            var sourceRow = source.getRow(i);
            int colNum = sourceRow == null ? -1 : sourceRow.getLastCellNum();
            if (colNum < 1) {
                // 没有有效的数据列
                continue;
            }
            var targetRow = RowUtil.getOrCreateRow(target, i);
            for (int j = sourceRow.getFirstCellNum(); j < colNum; j++) {
                var sourceCell = sourceRow.getCell(j);
                if (sourceCell == null) {
                    continue;
                }
                var targetCell = CellUtil.getOrCreateCell(targetRow, j);
                // 复制值
                targetCell.setCellValue(sourceCell.getStringCellValue());
                // 复制样式
                targetCell.setCellStyle(StyleUtil.cloneCellStyle(targetCell, sourceCell.getCellStyle()));
            }
        }
    }
}
