package com.elitescloud.boot.excel.config.tmpl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.excel.common.DataImport;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 数据导入服务工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/4
 */
@Slf4j
class DataImportServiceFactory {

    private final Map<String, ServiceMetaData> dataImportMap = new HashMap<>(64);

    public DataImportServiceFactory(List<DataImport<?>> dataImportList) {
        init(dataImportList);
    }

    /**
     * 判断是否有支持的导入服务
     *
     * @param tmplCode 模板编号
     * @return 是否有支持
     */
    public boolean isSupport(String tmplCode) {
        return dataImportMap.containsKey(tmplCode);
    }

    /**
     * 获取导入服务
     *
     * @param tmplCode 模板编号
     * @return 数据导入服务
     */
    public ServiceMetaData getDataImportService(String tmplCode) {
        return dataImportMap.get(tmplCode);
    }

    private void init(List<DataImport<?>> dataImportList) {
        if (CollUtil.isEmpty(dataImportList)) {
            return;
        }
        for (var dataImport : dataImportList) {
            if (dataImportMap.containsKey(dataImport.getTmplCode())) {
                throw new IllegalStateException("存在重复的数据导入服务：" + dataImport.getTmplCode());
            }
            dataImportMap.put(dataImport.getTmplCode(), new ServiceMetaData(dataImport));
        }
    }

    @Data
    public static class ServiceMetaData {
        private final DataImport<Serializable> dataImport;
        private final Class<?> dataType;

        public ServiceMetaData(DataImport<?> dataImport) {
            this.dataImport = (DataImport<Serializable>) dataImport;
            this.dataType = (Class<?>) ((ParameterizedType) dataImport.getClass().getGenericInterfaces()[0]).getActualTypeArguments()[0];
        }
    }
}
