package com.elitescloud.boot.excel.config.tmpl.export;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.excel.common.param.ImportRecordDTO;
import com.elitescloud.boot.excel.common.support.TmplApiService;
import com.elitescloud.boot.excel.common.support.TmplConstants;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import com.elitescloud.boot.util.LimiterUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.constraints.NotBlank;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/17/2023
 */
@Log4j2
public class SystemTmplDataSupport {

    private final TmplApiService tmplApiService;
    private final RedisUtils redisUtils;
    private final FileService<?> fileService;
    private final LimiterUtil limiterUtil;

    private TenantDataIsolateProvider tenantDataIsolateProvider;

    private static final String CACHE_KEY_IMPORT_RATE_PREFIX = "cloudt:tmpl_imp:rate:";

    @SuppressWarnings("unchecked")
    public SystemTmplDataSupport(TmplApiService tmplApiService, RedisUtils redisUtils, FileService<?> fileService) {
        this.tmplApiService = tmplApiService;
        this.redisUtils = redisUtils;
        this.fileService = fileService;
        this.limiterUtil = LimiterUtil.getInstance(redisUtils.getRedisTemplate(), "tmpl_data");
    }

    /**
     * 根据模板编号下载模板文件
     *
     * @param code 模板编号
     * @return 模板文件
     */
    public HttpEntity<StreamingResponseBody> downloadByCode(String code) {
        // 查询模板文件编号
        var tmplResult = tmplApiService.getFileCodeByCode(code);
        if (StringUtils.hasText(tmplResult.getData())) {
            return this.downloadByTmplFileCode(tmplResult.getData());
        }

        // 查询失败
        log.error("查询模板文件失败：{}", tmplResult);
        return ResponseEntity.notFound().build();
    }

    /**
     * 根据模板文件编号下载模板文件
     *
     * @param fileCode 模板文件编号
     * @return 模板文件
     */
    public HttpEntity<StreamingResponseBody> downloadByTmplFileCode(String fileCode) {
        if (fileService == null) {
            throw new BusinessException("文件服务未配置");
        }
        return fileService.download(fileCode, null);
    }

    /**
     * 根据模板编号获取模板信息
     *
     * @param code 模板编号
     * @return 模板信息
     */
    public SysTmplDTO getTmplByCode(@NotBlank String code) {
        // 先从缓存获取
        String cacheKey = TmplConstants.KEY_TMPL.replace("{TMPL_CODE}", code);
        SysTmplDTO tmplDTO = null;
        try {
            tmplDTO = this.redisDefault(() -> (SysTmplDTO) redisUtils.get(cacheKey));
        } catch (Exception e) {
            log.error("调用redis查询模板信息{}异常：", code, e);
        }
        if (tmplDTO != null) {
            return tmplDTO;
        }

        // 调用远程接口查询
        ApiResult<SysTmplDTO> result = tmplApiService.getByCode(code);
        if (result == null || !result.isSuccess()) {
            log.error("查询模板【{}】信息失败：{}", code, result);
            throw new BusinessException("查询模板信息失败");
        }

        // 放入缓存
        this.redisDefault(() -> redisUtils.set(cacheKey, result.getData()));
        return result.getData();
    }

    /**
     * 保存导入导出记录
     *
     * @param code       模板编号
     * @param dataFile   数据文件
     * @param queryParam 查询参数
     * @return 记录标识
     */
    public Long saveRecord(@NotBlank String code, MultipartFile dataFile, Map<String, Object> queryParam) {
        // 先上传文件
        String fileCode = null;
        if (dataFile != null) {
            if (fileService == null) {
                throw new BusinessException("文件服务未配置");
            }
            var uploadResult = fileService.upload(dataFile, Collections.emptyMap());
            if (!uploadResult.isSuccess() || uploadResult.getData() == null) {
                log.error("上传导入文件失败：{}", uploadResult);
                throw new BusinessException("请确认文件服务正常");
            }
            fileCode = uploadResult.getData().getFileCode();
        }

        // 保存记录
        var saveResult = tmplApiService.saveRecord(code, fileCode, ObjectUtil.defaultIfNull(queryParam, Collections.emptyMap()));
        if (!saveResult.isSuccess()) {
            log.error("保存模板【{}】导入导出数据记录失败：{}", code, saveResult);
            throw new BusinessException(saveResult.getMsg());
        }

        return saveResult.getData();
    }

    /**
     * 更新限流信息
     *
     * @param tmplDTO 模板信息
     * @param add     是否新增
     * @return 更新结果
     */
    public boolean updateLimiter(SysTmplDTO tmplDTO, boolean add) {
        if (tmplDTO.getConcurrentLimit() == null || tmplDTO.getConcurrentLimit() == -1) {
            // 不限流
            return true;
        }
        return limiterUtil.updateLimiter(tmplDTO.getConcurrentLimit(), 1, add);
    }

    /**
     * 更新导入数量
     *
     * @param recordId 导入记录标识
     * @param numTotal 导入总数量
     */
    public void updateImportNum(Long recordId, Long numTotal) {
        var result = tmplApiService.updateNumTotal(recordId, numTotal);
        if (result == null || !result.isSuccess()) {
            throw new BusinessException("更新操作记录失败");
        }
    }

    /**
     * 更新导入结果
     *
     * @param importResultDTO 导入结果
     */
    public void updateImportResult(RecordResultSaveDTO importResultDTO) {
        ApiResult<Long> result = tmplApiService.updateResult(importResultDTO);
        if (result == null || !result.isSuccess()) {
            throw new BusinessException("更新导入结果失败");
        }
    }

    /**
     * 保存导出文件
     *
     * @param recordId 导入记录标识
     * @param fileCode 导出文件编码
     * @param order    顺序
     * @return
     */
    public Long saveExportFile(Long recordId, String fileCode, int order) {
        var result = tmplApiService.saveExportFile(recordId, fileCode, order);
        if (result == null || !result.isSuccess()) {
            log.error("保存模板【{}, {}】导出数据记录失败：{}", recordId, order, result);
            throw new BusinessException("保存导出记录失败");
        }

        return result.getData();
    }

    /**
     * 保存导入失败的记录
     *
     * @param recordId 导入记录ID
     * @param fileCode 失败记录的文件编码
     */
    public void saveImportFailRecord(Long recordId, String fileCode) {
        if (recordId == null || CharSequenceUtil.isBlank(fileCode)) {
            return;
        }
        var res = tmplApiService.saveImportFailRecord(recordId, fileCode);
        if (res.isFailed()) {
            log.error("保存导入{}失败的记录{}失败：{}", recordId, fileCode, res.getMsg());
        }
    }

    /**
     * 缓存导入导出进度
     *
     * @param recordId 导入导出记录ID
     * @param rateDTO  导入进度
     */
    public void storeRate(Long recordId, SysImportRateDTO rateDTO) {
        var key = CACHE_KEY_IMPORT_RATE_PREFIX + recordId;
        redisUtils.set(key, rateDTO, Duration.ofMinutes(20).toSeconds());
    }

    /**
     * 移除导入导出进度
     *
     * @param recordId 导入导出记录ID
     */
    public void removeRate(Long recordId) {
        var key = CACHE_KEY_IMPORT_RATE_PREFIX + recordId;
        redisUtils.del(key);
    }

    /**
     * 获取导入进度
     *
     * @param recordId 导入记录ID
     * @return 导入进度
     */
    public SysImportRateDTO getImportRateFromCache(Long recordId) {
        return (SysImportRateDTO) redisUtils.get(CACHE_KEY_IMPORT_RATE_PREFIX + recordId);
    }

    /**
     * 获取导入进度
     *
     * @param recordId 导入记录ID
     * @return 导入进度
     */
    public SysImportRateDTO getImportRate(Long recordId) {
        var result = tmplApiService.getImportRate(recordId);
        if (result == null || !result.isSuccess()) {
            log.error("查询导入进度【{}】失败：{}", recordId, result);
            throw new BusinessException("查询导入进度失败");
        }

        return result.getData();
    }

    /**
     * 获取记录的文件编号
     *
     * @param recordId 导入记录ID
     * @return 导入进度
     */
    public String getRecordFileCode(Long recordId) {
        var result = tmplApiService.getRecordFileCode(recordId);
        if (result == null || !result.isSuccess()) {
            log.error("查询记录的文件编号【{}】失败：{}", recordId, result);
            throw new BusinessException("获取文件失败");
        }

        return result.getData();
    }

    /**
     * 获取未导入结束的
     *
     * @return 导入记录ID
     */
    public List<Long> getUnFinished() {
        var result = tmplApiService.queryUnFinished(null);
        if (result == null || !result.isSuccess()) {
            log.error("查询未导入结束的失败：{}", result);
            throw new BusinessException("查询未导入结束的失败");
        }

        return result.getData();
    }

    /**
     * 更新异常导致的结束
     *
     * @param failMsg 异常信息
     */
    public void updateResultForSysError(String failMsg) {
        String appCode = CloudtAppHolder.getAppCode();
        var res = tmplApiService.updateResultForError(appCode, failMsg);
        if (res.isFailed()) {
            log.error("更新导入异常结果失败：{}", res.getMsg());
        }
    }

    /**
     * 查询导入记录列表
     *
     * @param tmplCode 模板编码
     * @param hours    小时数量
     * @param self     是否只查自己的
     * @return 记录列表
     */
    public List<ImportRecordDTO> queryRecord(@NotBlank String tmplCode, Integer hours, Boolean self) {
        return tmplApiService.queryRecord(tmplCode, hours, self).computeData();
    }

    private <T> T redisDefault(Supplier<T> supplier) {
        return tenantDataIsolateProvider.byDefaultDirectly(supplier::get);
    }

    @Autowired
    public void setTenantDataIsolateProvider(TenantDataIsolateProvider tenantDataIsolateProvider) {
        this.tenantDataIsolateProvider = tenantDataIsolateProvider;
    }
}
