package com.elitescloud.boot.excel.config.tmpl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.excel.common.DataExport;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 数据导出服务工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/4
 */
@Slf4j
class DataExportServiceFactory {

    private final Map<String, ServiceMetaData> dataExportMap = new HashMap<>(64);

    public DataExportServiceFactory(List<DataExport<?, ?>> dataExportList) {
        init(dataExportList);
    }

    /**
     * 判断是否有支持的导出服务
     *
     * @param tmplCode 模板编号
     * @return 是否有支持
     */
    public boolean isSupport(String tmplCode) {
        return dataExportMap.containsKey(tmplCode);
    }

    /**
     * 获取导出服务
     *
     * @param tmplCode 模板编号
     * @return 数据导入服务
     */
    public ServiceMetaData getDataExportService(String tmplCode) {
        return dataExportMap.get(tmplCode);
    }

    private void init(List<DataExport<? extends Serializable, ? extends AbstractOrderQueryParam>> dataExportList) {
        if (CollUtil.isEmpty(dataExportList)) {
            return;
        }
        for (var dataExport : dataExportList) {
            if (dataExportMap.containsKey(dataExport.getTmplCode())) {
                throw new BusinessException("存在重复的数据导出服务：" + dataExport.getTmplCode());
            }
        }
        CompletableFuture.runAsync(() -> {
            for (var dataExport : dataExportList) {
                dataExportMap.put(dataExport.getTmplCode(), new ServiceMetaData(dataExport));
            }
        }).whenComplete((r, e) -> {
            if (e != null) {
                log.error("初始化数据导出服务异常：", e);
                return;
            }
            log.info("initialized DataExportService：{}", String.join(",", dataExportMap.keySet()));
        });
    }

    public static class ServiceMetaData {
        private final DataExport dataExport;
        private final Class<? extends Serializable> dataType;
        private final Class<? extends AbstractOrderQueryParam> paramType;
        private final boolean hasNewExportMethod;
        private final Method exportMethod;
        private final Method exportMethodCompatible;

        @SuppressWarnings("unchecked")
        public <T extends DataExport<R, P>, R extends Serializable, P extends AbstractOrderQueryParam> ServiceMetaData(T dataExport) {
            this.dataExport = dataExport;
            var clazzDataExport = dataExport.getClass();

            var arguments = ((ParameterizedType) clazzDataExport.getGenericInterfaces()[0]).getActualTypeArguments();
            this.dataType = (Class<R>) arguments[0];
            this.paramType = (Class<P>) arguments[1];

            // 寻找导出的执行方法
            this.exportMethod = this.obtainExportMethod(dataExport, paramType);
            this.exportMethodCompatible = this.obtainExportMethodCompatible(dataExport, paramType);
            Assert.isTrue(exportMethod != null || exportMethodCompatible != null, () -> dataExport.getClass().getName() + "未找到导出数据的executeExport方法");

            // 判断是否使用了新的导出方法
//            this.hasNewExportMethod = this.detectedHasNewExportMethod(clazzDataExport, paramType);
            this.hasNewExportMethod = exportMethod != null;
            if (!hasNewExportMethod) {
                log.warn("{}使用了过期的导出方法，请尽快切换成最新的导出方法executeExport()", clazzDataExport.getName());
            }
        }

        @SuppressWarnings(value = {"all"})
        public <R extends Serializable, P extends AbstractOrderQueryParam> PagingVO<R> applyExport(P params, boolean useReflect) {
            try {
                // 部分情况下直接使用对象的方法会使得调到接口中的默认方法而非实现类的方法，所以提供反射的方式调用
                if (useReflect) {
                    if (exportMethod != null) {
                        return (PagingVO<R>) exportMethod.invoke(dataExport, params);
                    }
                    if (exportMethodCompatible != null) {
                        return (PagingVO<R>) exportMethodCompatible.invoke(dataExport, params, params.getCurrent() + 1, params.getSize());
                    }
                }

                if (hasNewExportMethod) {
                    return (PagingVO<R>) dataExport.executeExport(params);
                }

                return ((PagingVO<R>) dataExport.execute(params, params.getCurrent() + 1, params.getSize()));
            } catch (Throwable e) {
                throw new BusinessException("导出数据异常", e);
            }
        }

        @SuppressWarnings("unchecked")
        public <P extends AbstractOrderQueryParam> Class<P> applyGetParamType() {
            return (Class<P>) paramType;
        }

        public String applyGetExportFileName() {
            return ((DataExport<?, ?>) dataExport).exportFileName();
        }

        public int applyGetPageSize() {
            var pageSize = ((DataExport<?, ?>) dataExport).pageSize();
            if (pageSize == null || pageSize < 1) {
                // 默认页大小
                return 50;
            }

            return pageSize;
        }

        public boolean applyIsRequiredTmpl() {
            return Boolean.TRUE.equals(((DataExport<?, ?>) dataExport).requireTmpl());
        }

        private boolean detectedHasNewExportMethod(Class<?> clazz, Class<?> paramType) {
            try {
                clazz.getDeclaredMethod("executeExport", paramType);
                return true;
            } catch (NoSuchMethodException e) {
                return false;
            }
        }

        private Method obtainExportMethod(Object dataExport, Class<?> paramType) {
            try {
                return dataExport.getClass().getDeclaredMethod("executeExport", paramType);
            } catch (Throwable ignored) {
            }
            return null;
        }

        private Method obtainExportMethodCompatible(Object dataExport, Class<?> paramType) {
            try {
                return dataExport.getClass().getDeclaredMethod("execute", paramType, int.class, int.class);
            } catch (Throwable ignored) {
            }
            return null;
        }
    }
}
