package com.elitescloud.boot.excel.config.tmpl.export.strategy;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.excel.common.ExcelConstant;
import com.elitescloud.boot.excel.config.tmpl.export.CostTime;
import com.elitescloud.boot.excel.config.tmpl.export.ExportCostTime;
import com.elitescloud.boot.excel.config.tmpl.export.ExportStrategyParam;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.boot.excel.util.ExcelUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.poi.ss.usermodel.Sheet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 导出策略的抽象服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/28
 */
abstract class BaseExportStrategy<R extends Serializable, P extends AbstractOrderQueryParam> implements ExportStrategy<R, P> {
    private static final Logger LOG = LoggerFactory.getLogger(BaseExportStrategy.class);

    protected static final Integer SHEET_ROW_MAX = ExcelConstant.SHEET_ROW_MAX;
    protected final FileService<?> fileService;
    protected final SystemTmplDataSupport fsmTmplSupport;
    protected final ObjectMapper objectMapper;

    protected BaseExportStrategy(FileService<?> fileService, SystemTmplDataSupport fsmTmplSupport, ObjectMapper objectMapper) {
        this.fileService = fileService;
        this.fsmTmplSupport = fsmTmplSupport;
        this.objectMapper = objectMapper;
    }

    /**
     * 表头
     *
     * @param tmplDTO 模板信息
     * @return 表头
     */
    protected List<List<String>> obtainHeaders(SysTmplDTO tmplDTO) {
        Assert.notEmpty(tmplDTO.getAttributes(), "未获取到模板的头部信息");
        return List.of(tmplDTO.getAttributeTitles());
    }

    /**
     * 表头
     *
     * @param tmplDTO
     * @return
     */
    protected List<String> obtainAttributes(SysTmplDTO tmplDTO) {
        Assert.notEmpty(tmplDTO.getAttributes(), "未获取到模板的头部信息");
        return tmplDTO.getAttributes().get(tmplDTO.getFieldTypeRow() - 1);
    }

    /**
     * 更新导出进度
     *
     * @param recordId 导入导出进度ID
     * @param rateDTO  进度
     */
    protected void updateRate(long recordId, SysImportRateDTO rateDTO) {
        fsmTmplSupport.storeRate(recordId, rateDTO);
    }

    /**
     * 更新导出完毕状态
     *
     * @param param
     * @param numSuc
     * @param file
     */
    protected void updateExportFinish(ExportStrategyParam<R, P> param, Long numSuc, File file, CostTime costTime) {
        Exception exp = null;
        // 更新导入结果
        if (param.getImportId() != null) {
            String fileCode = null;
            try {
                fileCode = uploadImportFile(file);
            } catch (Exception e) {
                exp = e;
            }
            var importResultDTO = RecordResultSaveDTO.builder()
                    .recordId(param.getImportId())
                    .success(exp == null)
                    .numSuc(numSuc)
                    .failMsg(exp == null ? "导出成功" : exp.getMessage())
                    .fileCode(fileCode)
                    .costTimeInfo(costTime == null ? null : costTime.shortSummary())
                    .build();
            fsmTmplSupport.updateImportResult(importResultDTO);

            // 删除缓存的进度
            fsmTmplSupport.removeRate(param.getImportId());
        }

        // 更新限流信息
        fsmTmplSupport.updateLimiter(param.getTmplDTO(), false);
    }

    /**
     * 上传导出的文件
     *
     * @param param
     * @param file
     * @param fileIndex
     */
    protected void updateExportFile(ExportStrategyParam<R, P> param, File file, int fileIndex) {
        String fileCode = this.uploadImportFile(file);
        fsmTmplSupport.saveExportFile(param.getImportId(), fileCode, fileIndex);
    }

    /**
     * 转换并往excel写入数据
     *
     * @param excelWriter
     * @param sheet
     * @param attributes
     * @param tempDataList
     * @param costTime
     */
    protected void convertAndWrite(ExcelWriter excelWriter, WriteSheet sheet, List<String> attributes,
                                   List<R> tempDataList, ExportCostTime costTime) {
        // 数据转换
        costTime.getConvertData().start();
        var writeDataList = convertExportData(tempDataList, attributes);
        costTime.getConvertData().stop();

        // 写入数据
        costTime.getWriteData().start();
        excelWriter.write(writeDataList, sheet);
        costTime.getWriteData().stop();
    }

    /**
     * 将业务返回的数据转为字符串列表
     *
     * @param dataList
     * @param fields
     * @return
     */
    protected List<List<Object>> convertExportData(List<R> dataList, List<String> fields) {
        var dataType = CollUtil.isEmpty(dataList) ? null : dataList.get(0).getClass();
        return objectMapper.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
                }).stream()
                .map(data -> fields.stream().map(field -> getValue(data, field, dataType)).collect(Collectors.toList()))
                .collect(Collectors.toList());
    }

    private String uploadImportFile(File file) {
        if (file != null) {
            if (file.length() == 0) {
                LOG.error("上传导入导出结果文件异常，文件{}为空！", file.getAbsolutePath());
                return null;
            }
            var uploadResult = fileService.upload(file);
            // 删除临时文件
            file.delete();
            if (uploadResult.isSuccess()) {
                return uploadResult.getData().getFileCode();
            }
            LOG.error("上传导入导出结果文件失败：{}", uploadResult);
            throw new BusinessException("文件服务异常");
        }
        return null;
    }

    private Object getValue(Map<String, Object> data, String field, Class<?> dataType) {
        return ExcelUtil.formatExportedValue(data.get(field), field, dataType);
    }

    /**
     * 创建excel writer
     *
     * @param file excel文件
     * @return
     */
    protected ExcelWriter createExcelWriter(File file) {
        return ExcelUtil.normalizeExcelWriter(EasyExcel.write(file)).build();
    }

    /**
     * 创建导出临时文件
     *
     * @return
     */
    protected File createExportFile(ExportStrategyParam<R, P> param) {
        return createExportFile(param, null);
    }

    /**
     * 创建导出临时文件
     *
     * @param param
     * @param order
     * @return
     */
    protected File createExportFile(ExportStrategyParam<R, P> param, Integer order) {
        String fileName = param.getExportFileName();
        if (order != null) {
            fileName = fileName + "_" + order;
        }

        var fileDir = new File(System.getProperty("java.io.tmpdir") + File.separator + LocalDateTime.now().format(DatetimeUtil.FORMATTER_DATETIME_LONG));
        if (!fileDir.exists() && !fileDir.mkdirs()) {
            throw new IllegalStateException("导出异常，创建临时文件夹" + fileDir.getAbsolutePath() + "失败");
        }

        var file = new File(fileDir, fileName + ".xlsx");
        file.delete();
        try {
            file.createNewFile();
        } catch (IOException e) {
            throw new RuntimeException("创建临时文件失败", e);
        }
        return file;
    }

    /**
     * 获取sheet的数据量限制
     *
     * @param tmplDTO
     * @return
     */
    protected int sheetLimit(SysTmplDTO tmplDTO) {
        Integer limit = tmplDTO.getExportSheetLimit();
        if (limit == null) {
            return SHEET_ROW_MAX;
        }
        if (limit > 0 && limit <= SHEET_ROW_MAX) {
            return limit;
        }

        throw new IllegalArgumentException(tmplDTO.getCode() + "参数" + limit + "不合法");
    }
}
