package com.elitescloud.boot.excel.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.elitescloud.boot.excel.common.ExcelConstant;
import com.elitescloud.boot.excel.common.param.ExportColumnParam;
import com.elitescloud.boot.util.ObjectMapperFactory;
import com.elitescloud.cloudt.context.util.DatetimeUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

/**
 * excel导出工具类.
 * <p>
 * 基于EasyExcel实现的工具类，
 * 启用，请参考使用ExportExcelService
 *
 * @author Kaiser（wang shao）
 * @date 2020-12-31
 * @see <a href="https://www.yuque.com/easyexcel/doc/easyexcel">sss</a>
 */
public class ExcelExportUtil {
    private final Logger logger = LoggerFactory.getLogger(ExcelExportUtil.class);

    /**
     * 导出限制
     */
    public static final String PROPERTY_EXPORT_LIMIT = ExcelConstant.PROPERTY_EXPORT_LIMIT;

    private final ExcelWriterBuilder builder;
    private List<String> fields;
    /**
     * 每次最多写入50条
     */
    private Integer size = 50;

    protected ExcelExportUtil(OutputStream outputStream) {
        builder = EasyExcelFactory.write(outputStream)
                .autoCloseStream(true)
        ;
    }

    /**
     * 导出工具实例化
     *
     * @param outputStream 文件输出流
     * @return 工具实例
     * @throws IOException 异常
     */
    public static ExcelExportUtil instance(OutputStream outputStream) throws IOException {
        return new ExcelExportUtil(outputStream);
    }

    /**
     * 数据的标题
     *
     * @param columnParamList 导出的列
     * @return 异常
     */
    public ExcelExportUtil fields(List<ExportColumnParam> columnParamList) {
        if (columnParamList != null && !columnParamList.isEmpty()) {
            List<List<String>> titles = new ArrayList<>(64);
            List<String> includeFields = new ArrayList<>(64);
            for (ExportColumnParam p : columnParamList) {
                if (StrUtil.isBlank(p.getTitle()) || StrUtil.isBlank(p.getField())) {
                    continue;
                }
                titles.add(List.of(p.getTitle()));
                includeFields.add(p.getField());
            }
            this.builder.head(titles);
            this.fields = includeFields;
        }
        return this;
    }

    /**
     * 分页获取时每次获取的数据量
     *
     * @param size 数据量
     * @return 工具实例
     */
    public ExcelExportUtil batchSize(Integer size) {
        if (size == null || size < 1) {
            throw new IllegalArgumentException("size参数不合法");
        }

        this.size = size;
        return this;
    }

    /**
     * 日期格式
     *
     * @param dateFormat 日期格式
     * @return 工具实例
     */
    @Deprecated(forRemoval = true, since = "3.2.4")
    public ExcelExportUtil dateFormat(String dateFormat) {
        return this;
    }

    /**
     * 开始写数据
     *
     * @param sheetName sheet名称
     * @param data      数据
     */
    public void write(String sheetName, List<?> data) {
        long start = System.currentTimeMillis();
        Assert.notEmpty(fields, "未知导出的数据列");

        List<List<String>> dataConvert = convertAllData(data);

        this.builder.sheet(sheetName).doWrite(dataConvert);

        logger.info("excel导出完毕，用时：{}ms", System.currentTimeMillis() - start);
    }

    /**
     * 开始写数据
     *
     * @param sheetName    sheet名称
     * @param dataProducer 数据生产者
     */
    public void write(String sheetName, BiFunction<Integer, Integer, List<?>> dataProducer) {
        long start = System.currentTimeMillis();
        Assert.notEmpty(fields, "未知导出的数据列");

        var excelWriter = this.builder.build();

        try {
            writeDataByPage(excelWriter, sheetName, dataProducer);
        } catch (Exception e) {
            logger.error("导出数据异常：", e);
        } finally {
            excelWriter.finish();
        }

        logger.info("excel导出完毕，用时：{}ms", System.currentTimeMillis() - start);
    }

    private void writeDataByPage(ExcelWriter excelWriter, String sheetName, BiFunction<Integer, Integer, List<?>> dataProducer) {
        // 获取导出数量限制
        Integer limit = getLimitNumOfExport();
        logger.info("导出数量限制：{}", limit);
        if (limit != null && limit < 1) {
            this.builder.sheet(sheetName).doWrite(Collections.emptyList());
            return;
        }

        var writerSheetBuilder = new ExcelWriterSheetBuilder(excelWriter);
        writerSheetBuilder.sheetName(sheetName);
        var writeSheet = writerSheetBuilder.build();

        var rows = 0;
        var page = 1;
        List<?> dataList = null;
        var objectMapper = getObjectMapper();
        // 分页导出数据
        while (true) {
            dataList = dataProducer.apply(page++, size);
            if (CollectionUtils.isEmpty(dataList)) {
                if (page == 2) {
                    excelWriter.write(Collections.emptyList(), writeSheet);
                }
                break;
            }

            rows += dataList.size();
            if (limit != null && rows > limit) {
                // 超出限制
                dataList = CollectionUtil.sub(dataList, 0, limit - rows + size);
            }

            // 转换数据格式后写入excel
            List<List<String>> convertData = objectMapper.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
            }).stream().map(data ->
                    fields.stream().map(field -> obtainValue(data, field)
                    ).collect(Collectors.toList())
            ).collect(Collectors.toList());
            excelWriter.write(convertData, writeSheet);

            if (dataList.size() < size) {
                // 少于获取的数量，则说明已无数据
                break;
            }
        }
    }

    /**
     * 转换数据格式
     *
     * @param dataList 数据
     * @return 转换后的数据
     */
    private List<List<String>> convertAllData(List<?> dataList) {
        if (CollectionUtils.isEmpty(dataList)) {
            return Collections.emptyList();
        }

        // 获取导出数量限制
        Integer limit = getLimitNumOfExport();
        logger.info("导出数量限制：{}", limit);
        if (limit != null) {
            if (limit < 1) {
                return Collections.emptyList();
            }
            dataList = CollectionUtil.sub(dataList, 0, limit);
        }

        return convertData(dataList);
    }

    /**
     * 转换数据格式
     *
     * @param dataList 数据
     * @return 转换后的数据
     */
    private List<List<String>> convertData(List<?> dataList) {
        return getObjectMapper().convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
        }).stream().map(data ->
                fields.stream().map(field -> obtainValue(data, field)).collect(Collectors.toList())
        ).collect(Collectors.toList());
    }

    /**
     * 获取导出数量的限制
     *
     * @return 导出的数量限制
     */
    private Integer getLimitNumOfExport() {
        var limit = System.getProperty(PROPERTY_EXPORT_LIMIT, "1000000");
        return limit.equals("-1") ? null : Integer.valueOf(limit);
    }

    private String obtainValue(Map<String, Object> data, String field) {
        Object value = data.get(field);
        if (value == null) {
            return "";
        }
        if (value instanceof LocalDateTime) {
            return DatetimeUtil.FORMATTER_DATETIME.format((LocalDateTime) value);
        } else if (value instanceof Double) {
            return NumberUtil.decimalFormatMoney((Double) value);
        } else if (value instanceof Float) {
            return NumberUtil.decimalFormatMoney((Float) value);
        }
        return value.toString();
    }

    private ObjectMapper getObjectMapper() {
        return ObjectMapperFactory.instance();
    }
}
