package com.elitescloud.boot.excel.util;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.builder.ExcelReaderBuilder;
import com.elitescloud.boot.excel.common.param.ImportDataModel;
import com.elitescloud.boot.excel.support.DataReadListener;

import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * 数据导入工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2021-03-16
 */
public class ExcelImportUtil {

    private final ExcelReaderBuilder excelReaderBuilder;
    private DataReadListener readListener;
    private Set<Integer> sheetNo;

    protected ExcelImportUtil(InputStream inputStream) {
        excelReaderBuilder = EasyExcelFactory.read(inputStream)
                .autoCloseStream(true)
                .autoTrim(true)
                .ignoreEmptyRow(true)
        ;
    }

    public static ExcelImportUtil instance(InputStream inputStream) {
        return new ExcelImportUtil(inputStream);
    }

    /**
     * 头部数据行数
     *
     * @param headRow 头部数据行数
     * @return 类
     */
    public ExcelImportUtil headRow(Integer headRow) {
        excelReaderBuilder.headRowNumber(headRow);
        return this;
    }

    /**
     * 要读取的sheet页数
     *
     * @param sheetNo sheetNo列表，从1开始
     * @return 类
     */
    public ExcelImportUtil sheetNo(Set<Integer> sheetNo) {
        this.sheetNo = sheetNo;
        return this;
    }

    /**
     * 设置数据类型
     *
     * @param dataType     数据类型
     * @param dataTypeAttr 数据类型类的属性，按excel中列的顺序
     * @return 类
     */
    public ExcelImportUtil dataType(Class<?> dataType, List<String> dataTypeAttr) {
        readListener = new DataReadListener(dataTypeAttr);
        readListener.setDataType(dataType);
        return this;
    }

    /**
     * 设置数据类型
     *
     * @param dataType     数据类型
     * @param dataTypeAttr 数据类型类的属性所在行，必须在headRow内
     * @return 类
     */
    public ExcelImportUtil dataType(Class<?> dataType, Integer dataTypeAttr) {
        readListener = new DataReadListener(dataTypeAttr);
        readListener.setDataType(dataType);
        return this;
    }

    /**
     * 同步读取所有数据
     * <p>
     * 数据量大时不建议
     *
     * @return 所有数据
     */
    public List<?> readAllSync() throws Exception {
        if (readListener == null && CollUtil.isEmpty(sheetNo)) {
            return excelReaderBuilder.doReadAllSync();
        }

        if (readListener == null) {
            readListener = new DataReadListener();
        }
        readListener.setPredicateReadRow(this.buildReadPredicate());
        excelReaderBuilder.registerReadListener(readListener);

        CompletableFuture<Boolean> completableFuture = CompletableFuture.supplyAsync(() -> {
            excelReaderBuilder.doReadAll();
            return readListener.isFinish();
        });
        if (Boolean.TRUE.equals(completableFuture.get())) {
            return readListener.getDataList();
        }

        return Collections.emptyList();
    }

    /**
     * 异步读取数据
     *
     * @param dataConsumer 数据消费者
     * @throws Exception 异常
     */
    public void readAllAsync(Consumer<ImportDataModel> dataConsumer) throws Exception {
        readAllAsync(dataConsumer, null);
    }

    /**
     * 异步读取数据
     *
     * @param dataConsumer 数据消费者
     * @param batchSize    每次读取的数据量，如果为空或不大于1，则表示无限制，一次性获取
     * @throws Exception 异常
     */
    public void readAllAsync(Consumer<ImportDataModel> dataConsumer, Integer batchSize) throws Exception {
        if (readListener == null) {
            readListener = new DataReadListener();
        }
        readListener.setDataSize(batchSize);
        readListener.setDataConsumer(dataConsumer);
        readListener.setPredicateReadRow(this.buildReadPredicate());
        excelReaderBuilder.registerReadListener(readListener);

        excelReaderBuilder.doReadAll();
    }

    private Predicate<AnalysisContext> buildReadPredicate() {
        if (CollUtil.isEmpty(sheetNo)) {
            return t -> true;
        }

        return t -> sheetNo.contains(t.readSheetHolder().getSheetNo() + 1);
    }
}
