package com.elitescloud.boot.excel.config.tmpl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

/**
 * 数据导入服务工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/4
 */
@Slf4j
class DataImportServiceFactory {

    private final Map<String, ServiceMetaData> dataImportMap = new HashMap<>(64);

    public DataImportServiceFactory(List<DataImport<? /*extends Serializable*/>> dataImportList) {
        init(dataImportList);
    }

    /**
     * 判断是否有支持的导入服务
     *
     * @param tmplCode 模板编号
     * @return 是否有支持
     */
    public boolean isSupport(String tmplCode) {
        return dataImportMap.containsKey(tmplCode);
    }

    /**
     * 获取导入服务
     *
     * @param tmplCode 模板编号
     * @return 数据导入服务
     */
    public ServiceMetaData getDataImportService(String tmplCode) {
        return dataImportMap.get(tmplCode);
    }

    private void init(List<DataImport<? /*extends Serializable*/>> dataImportList) {
        if (CollUtil.isEmpty(dataImportList)) {
            return;
        }
        for (var dataImport : dataImportList) {
            if (dataImportMap.containsKey(dataImport.getTmplCode())) {
                throw new IllegalStateException("存在重复的数据导入服务：" + dataImport.getTmplCode());
            }
        }
        CompletableFuture.runAsync(() -> {
            for (var dataImport : dataImportList) {
                dataImportMap.put(dataImport.getTmplCode(), new ServiceMetaData(dataImport));
            }
        }).whenComplete((r, e) -> {
            if (e != null) {
                log.error("初始化数据导入服务异常：", e);
                return;
            }
            log.info("initialized DataImportService：{}", String.join(",", dataImportMap.keySet()));
        });
    }

    public static class ServiceMetaData {
        private final String tmplCode;
        private final Object dataImport;
        private final Class<?/* extends Serializable*/> dataType;
        private final Method importMethod;
        private final Method beforeImportMethod;
        private final Method afterImportMethod;
        private final boolean respSuccessData;
        private final boolean respFailData;
        private final boolean respFailMsg;

        public <T extends DataImport<R>, /*R extends Serializable*/ R extends Object> ServiceMetaData(T dataImport) {
            this.tmplCode = dataImport.getTmplCode();
            this.dataImport = dataImport;

            // 参数类型
            var beanClass = DataImport.class.isAssignableFrom(dataImport.getClass().getSuperclass()) ?
                    dataImport.getClass().getSuperclass() : dataImport.getClass();
            this.dataType = (Class<?/* extends Serializable*/>) ((ParameterizedType) beanClass.getGenericInterfaces()[0]).getActualTypeArguments()[0];

            if (!Serializable.class.isAssignableFrom(dataType)) {
                log.error("参数类型：{}未实现序列化接口Serializable", dataType.getName());
            }
            try {
                this.importMethod = dataImport.getClass().getMethod("executeImport", List.class, int.class);
            } catch (NoSuchMethodException e) {
                throw new IllegalStateException("未获取到导入方法", e);
            }
            try {
                this.beforeImportMethod = dataImport.getClass().getMethod("beforeImport", List.class);
            } catch (NoSuchMethodException e) {
                throw new IllegalStateException("未获取到预导入处理方法", e);
            }
            try {
                this.afterImportMethod = dataImport.getClass().getMethod("afterImport", boolean.class);
            } catch (NoSuchMethodException e) {
                throw new IllegalStateException("未获取到后置导入处理方法", e);
            }
            this.respSuccessData = dataImport.respSuccessData();
            this.respFailData = dataImport.respFailData();
            this.respFailMsg = dataImport.respFailMsg();
        }

        public String applyGetTmplCode() {
            return tmplCode;
        }

        public Class<?> applyGetDataType() {
            return dataType;
        }

        public int applyGetStepSize() {
            Integer stepSize = ((DataImport<?>) dataImport).stepSize();
            if (stepSize == null || stepSize < 1) {
                // 默认页大小
                return 10;
            }

            return stepSize;
        }

        public Set<Integer> applyGetSheetNos() {
            return ((DataImport<?>) dataImport).sheetNoList();
        }

        public List<String> applyImport(List<? extends Serializable> dataList, int startRowIndex) {
            try {
                return (List<String>) importMethod.invoke(dataImport, dataList, startRowIndex);
            } catch (Throwable e) {
                throw new BusinessException("导入数据异常", e);
            }
        }

        public void applyBeforeImport(List<? extends Serializable> dataList) {
            try {
                beforeImportMethod.invoke(dataImport, dataList);
            } catch (Throwable e) {
                throw new BusinessException("导入失败，" + e.getMessage(), e);
            }
        }

        public void applyAfterImport(boolean sync) {
            try {
                afterImportMethod.invoke(dataImport, sync);
            } catch (Throwable e) {
                log.error("处理导入后置异常：", e);
            }
        }

        public boolean isRespSuccessData() {
            return respSuccessData;
        }

        public boolean isRespFailData() {
            return respFailData;
        }

        public boolean isRespFailMsg() {
            return respFailMsg;
        }
    }
}
