package com.elitescloud.boot.excel.config.tmpl.export.strategy;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.excel.common.ExcelConstant;
import com.elitescloud.boot.excel.config.tmpl.export.CostTime;
import com.elitescloud.boot.excel.config.tmpl.export.ExportCostTime;
import com.elitescloud.boot.excel.config.tmpl.export.ExportStrategyParam;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.boot.excel.support.TmplDataExportWrapper;
import com.elitescloud.boot.excel.util.ExcelUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.poi.ss.usermodel.CellType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 导出策略的抽象服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/28
 */
abstract class BaseExportStrategy<R extends Serializable, P extends AbstractOrderQueryParam> implements ExportStrategy<R, P> {
    private static final Logger LOG = LoggerFactory.getLogger(BaseExportStrategy.class);

    protected static final Integer SHEET_ROW_MAX = ExcelConstant.SHEET_ROW_MAX;
    protected final FileService<?> fileService;
    protected final SystemTmplDataSupport fsmTmplSupport;
    protected final ObjectMapper objectMapper;

    protected BaseExportStrategy(FileService<?> fileService, SystemTmplDataSupport fsmTmplSupport, ObjectMapper objectMapper) {
        this.fileService = fileService;
        this.fsmTmplSupport = fsmTmplSupport;
        this.objectMapper = objectMapper;
    }

    /**
     * 更新导出进度
     *
     * @param recordId 导入导出进度ID
     * @param rateDTO  进度
     */
    protected void updateRate(long recordId, SysImportRateDTO rateDTO) {
        fsmTmplSupport.storeRate(recordId, rateDTO);
    }

    /**
     * 更新导出完毕状态
     *
     * @param param
     * @param numSuc
     * @param file
     */
    protected void updateExportFinish(ExportStrategyParam<R, P> param, Long numSuc, File file, CostTime costTime) {
        Exception exp = null;
        // 更新导入结果
        if (param.getImportId() != null) {
            String fileCode = null;
            try {
                fileCode = uploadImportFile(file);
            } catch (Exception e) {
                exp = e;
            }
            var importResultDTO = RecordResultSaveDTO.builder()
                    .recordId(param.getImportId())
                    .success(exp == null)
                    .numSuc(numSuc)
                    .failMsg(exp == null ? "导出成功" : exp.getMessage())
                    .fileCode(fileCode)
                    .costTimeInfo(costTime == null ? null : costTime.shortSummary())
                    .build();
            fsmTmplSupport.updateImportResult(importResultDTO);

            // 删除缓存的进度
            fsmTmplSupport.removeRate(param.getImportId());
        }

        // 更新限流信息
        fsmTmplSupport.updateLimiter(param.getTmplDTO(), false);
    }

    /**
     * 上传导出的文件
     *
     * @param param
     * @param file
     * @param fileIndex
     */
    protected void updateExportFile(ExportStrategyParam<R, P> param, File file, int fileIndex) {
        String fileCode = this.uploadImportFile(file);
        fsmTmplSupport.saveExportFile(param.getImportId(), fileCode, fileIndex);
    }

    /**
     * 转换并往excel写入数据
     *
     * @param excelWriter
     * @param sheet
     * @param tmplWrapper
     * @param tempDataList
     * @param costTime
     */
    protected void convertAndWrite(ExcelWriter excelWriter, WriteSheet sheet, TmplDataExportWrapper tmplWrapper,
                                   List<R> tempDataList, ExportCostTime costTime) {
        // 数据转换
        costTime.getConvertData().start();
        var writeDataList = convertExportData(tempDataList, tmplWrapper);
        costTime.getConvertData().stop();

        // 写入数据
        costTime.getWriteData().start();
        excelWriter.write(writeDataList, sheet);
        costTime.getWriteData().stop();
    }

    /**
     * 将业务返回的数据转为字符串列表
     *
     * @param dataList
     * @param tmplWrapper
     * @return
     */
    private List<List<Object>> convertExportData(List<R> dataList, TmplDataExportWrapper tmplWrapper) {
        var dataType = CollUtil.isEmpty(dataList) ? null : dataList.get(0).getClass();
        var fields = tmplWrapper.getTmplDTO().getAttributeFields();
        return objectMapper.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
                }).stream()
                .map(data -> fields.stream().map(field -> getValue(data, field, dataType)).collect(Collectors.toList()))
                .collect(Collectors.toList());
    }

    private String uploadImportFile(File file) {
        if (file != null) {
            if (file.length() == 0) {
                LOG.error("上传导入导出结果文件异常，文件{}为空！", file.getAbsolutePath());
                return null;
            }
            var uploadResult = fileService.upload(file);
            // 删除临时文件
            file.delete();
            if (uploadResult.isSuccess()) {
                return uploadResult.getData().getFileCode();
            }
            LOG.error("上传导入导出结果文件失败：{}", uploadResult);
            throw new BusinessException("文件服务异常");
        }
        return null;
    }

    private Object getValue(Map<String, Object> data, String field, Class<?> dataType) {
        Object value = data.get(field);
        if (value == null && field.startsWith(ExcelConstant.EXTENSION_FIELD_PREFIX)) {
            // 尝试获取扩展字段
            Object extensionInfo = data.get(ExcelConstant.EXTENSION_PROPERTY);
            if (extensionInfo instanceof Map) {
                value = ((Map<String, Object>) extensionInfo).get(field.substring(ExcelConstant.EXTENSION_FIELD_PREFIX.length()));
            }
        }
        return ExcelUtil.formatExportedValue(value, field, dataType);
    }

    /**
     * 创建excel writer
     *
     * @param file excel文件
     * @return
     */
    protected ExcelWriter createExcelWriter(File file) {
        return ExcelUtil.normalizeExcelWriter(EasyExcel.write(file)).build();
    }

    protected WriteSheet createSheet(int sheetIndex, TmplDataExportWrapper tmplWrapper) {
        var sheetBuilder = EasyExcel.writerSheet(sheetIndex)
                .sheetName("Sheet " + (sheetIndex + 1));

        if (tmplWrapper.isFieldFromFront()) {
            // 来自前端，则直接写入头
            sheetBuilder.head(tmplWrapper.getTmplDTO().getAttributeTitles().stream().map(List::of).collect(Collectors.toList()));
        } else {
            // 来自模板，则在最后从模板上复制
            sheetBuilder.relativeHeadRowIndex(tmplWrapper.getTmplDTO().getHeadRow() - 1);
        }

        return sheetBuilder.build();
    }

    protected void copyHeaders(File file, int sheetNum, TmplDataExportWrapper tmplWrapper) throws Exception {
        if (tmplWrapper.getTmplSheet() == null) {
            return;
        }

        try (
                FileInputStream failFileInputStream = new FileInputStream(file);
        ) {
            var workBook = cn.hutool.poi.excel.ExcelUtil.getReader(failFileInputStream).getWorkbook();
            for (int i = 0; i < sheetNum; i++) {
                var sheet = workBook.getSheetAt(i);
                ExcelUtil.copyRows(tmplWrapper.getTmplSheet(), sheet, 0, tmplWrapper.getTmplDTO().getHeadRow() - 1, Set.of(tmplWrapper.getTmplDTO().getFieldTypeRow() - 1));

                // 扩展字段
                if (CollUtil.isNotEmpty(tmplWrapper.getExtensionField())) {
                    var row = sheet.getRow(0);
                    int colIndex = tmplWrapper.getTmplDTO().getAttributeFields().size() - tmplWrapper.getExtensionField().size();
                    var sourceCell = row.getCell(colIndex - 1);
                    for (String title : tmplWrapper.getExtensionField().values()) {
                        var cell = row.getCell(colIndex);
                        if (cell == null) {
                            cell = row.createCell(colIndex);
                        }
                        ExcelUtil.copyCell(sourceCell, cell);
                        cell.setCellType(CellType.STRING);
                        cell.setCellValue(title);
                        colIndex++;
                    }
                }
            }

            try (FileOutputStream failFileOutStream = new FileOutputStream(file)) {
                workBook.write(failFileOutStream);
                workBook.close();
            }
        }
    }

    /**
     * 创建导出临时文件
     *
     * @return
     */
    protected File createExportFile(ExportStrategyParam<R, P> param) {
        return createExportFile(param, null);
    }

    /**
     * 创建导出临时文件
     *
     * @param param
     * @param order
     * @return
     */
    protected File createExportFile(ExportStrategyParam<R, P> param, Integer order) {
        String fileName = param.getExportFileName();
        if (order != null) {
            fileName = fileName + "_" + order;
        }

        var fileDir = new File(System.getProperty("java.io.tmpdir") + File.separator + LocalDateTime.now().format(DatetimeUtil.FORMATTER_DATETIME_LONG));
        if (!fileDir.exists() && !fileDir.mkdirs()) {
            throw new IllegalStateException("导出异常，创建临时文件夹" + fileDir.getAbsolutePath() + "失败");
        }

        var file = new File(fileDir, fileName + ".xlsx");
        file.delete();
        try {
            file.createNewFile();
        } catch (IOException e) {
            throw new RuntimeException("创建临时文件失败", e);
        }
        return file;
    }

    /**
     * 获取sheet的数据量限制
     *
     * @param tmplDTO
     * @return
     */
    protected int sheetLimit(SysTmplDTO tmplDTO) {
        Integer limit = tmplDTO.getExportSheetLimit();
        if (limit == null) {
            return SHEET_ROW_MAX;
        }
        if (limit > 0 && limit <= SHEET_ROW_MAX) {
            return limit;
        }

        throw new IllegalArgumentException(tmplDTO.getCode() + "参数" + limit + "不合法");
    }
}
