package com.elitescloud.boot.excel.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.excel.common.ExcelConstant;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.boot.excel.util.ExcelUtil;
import com.elitescloud.boot.util.ArrayUtil;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Sheet;

import java.io.ByteArrayInputStream;
import java.util.*;

/**
 * 模板数据信息.
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/22
 */
@Slf4j
public class TmplDataExportWrapper {
    /**
     * 模板信息
     */
    private final SysTmplDTO tmplDTO;
    /**
     * 查询参数
     */
    private final Map<String, Object> queryParams;

    /**
     * 导出的扩展字段
     */
    private Map<String, String> extensionField;

    /**
     * 导出的字段是否来自前端
     */
    private boolean fieldFromFront;

    /**
     * 模板的sheet页
     */
    private Sheet tmplSheet;

    public TmplDataExportWrapper(SysTmplDTO tmplDTO, Map<String, Object> queryParams, SystemTmplDataSupport tmplDataSupport) {
        this.tmplDTO = tmplDTO;
        this.queryParams = queryParams;

        this.analyze(tmplDataSupport);
    }

    public SysTmplDTO getTmplDTO() {
        return tmplDTO;
    }

    public Map<String, Object> getQueryParams() {
        return queryParams;
    }

    public Map<String, String> getExtensionField() {
        return extensionField;
    }

    public boolean isFieldFromFront() {
        return fieldFromFront;
    }

    public Sheet getTmplSheet() {
        return tmplSheet;
    }

    private void analyze(SystemTmplDataSupport tmplDataSupport) {
        if (CollUtil.isEmpty(queryParams)) {
            return;
        }

        // 自定义导出的列
        this.dealExportColumn();

        // 扩展字段
        this.dealExtensionColumn();

        this.queryTmplSheet(tmplDataSupport);
    }

    private void dealExtensionColumn() {
        var extensionColumn = queryParams.get(ExcelConstant.PARAM_EXPORT_FIELD_EXTENSION);
        if (extensionColumn == null) {
            this.extensionField = Collections.emptyMap();
            return;
        }
        if (!(extensionColumn instanceof Collection)) {
            throw new IllegalArgumentException("扩展字段设置有误");
        }

        this.extensionField = new LinkedHashMap<>(((Collection<?>)extensionColumn).size());
        List<String> fields = new ArrayList<>(tmplDTO.getAttributeFields());
        List<String> titles = new ArrayList<>(tmplDTO.getAttributeTitles());
        Object field = null, title = null;
        String fieldName = null;
        for (Object obj : (Collection<?>) extensionColumn) {
            if (!(obj instanceof Map)) {
                throw new IllegalArgumentException("扩展字段的列设置有误");
            }
            var columnParam = (Map<?, ?>) obj;
            field = columnParam.get("field");
            title = columnParam.get("title");
            if (field == null) {
                continue;
            }
            if (!(field instanceof String)) {
                throw new IllegalArgumentException("扩展列设置有误");
            }

            fieldName = ExcelConstant.EXTENSION_FIELD_PREFIX + field;
            extensionField.put(fieldName, title == null ? "" : title.toString());

            fields.add(fieldName);
            titles.add(title == null ? "" : title.toString());
        }

        tmplDTO.setAttributeFields(fields);
        tmplDTO.setAttributeTitles(titles);
        tmplDTO.setAttributes(List.of(titles, fields));
    }

    private void dealExportColumn() {
        var exportColumn = queryParams.get(ExcelConstant.PARAM_EXPORT_FIELD);
        if (exportColumn == null) {
            return;
        }
        if (!(exportColumn instanceof Collection)) {
            throw new IllegalArgumentException("导出列设置有误");
        }

        List<String> fieldList = new ArrayList<>();
        List<String> titleList = new ArrayList<>();
        Object field = null, title = null;
        for (Object obj : (Collection<?>) exportColumn) {
            if (!(obj instanceof Map)) {
                throw new IllegalArgumentException("导出列设置有误");
            }
            var columnParam = (Map<?, ?>) obj;
            field = columnParam.get("field");
            title = columnParam.get("title");
            if (field == null) {
                continue;
            }
            if (!(field instanceof String)) {
                throw new IllegalArgumentException("导出列设置有误");
            }
            fieldList.add((String) field);
            titleList.add(title == null ? "" : title.toString());
        }

        tmplDTO.setAttributeFields(fieldList);
        tmplDTO.setAttributeTitles(titleList);
        tmplDTO.setAttributes(List.of(titleList, fieldList));
        tmplDTO.setHeadRow(2);
        tmplDTO.setFieldTypeRow(2);

        // 使用前端传的字段
        this.fieldFromFront = true;
    }

    private void queryTmplSheet(SystemTmplDataSupport tmplDataSupport) {
        if (this.isFieldFromFront()) {
            return;
        }

        var downloadResult = tmplDataSupport.getTmplFileByCode(tmplDTO.getCode());
        if (downloadResult.getData() == null) {
            log.error("获取模板文件失败：{}, {}", tmplDTO.getCode(), downloadResult.getMsg());
            return;
        }
        if (ArrayUtil.isEmpty(downloadResult.getData().getBytes())) {
            return;
        }
        this.tmplSheet = ExcelUtil.getReader(new ByteArrayInputStream(downloadResult.getData().getBytes()), 0).getSheet();
    }
}
